<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';
$generated = 0;

/**
 * Calculate prayer times based on coordinates
 * Using simplified calculation method for Bangladesh
 */
function calculatePrayerTimes($latitude, $longitude, $date) {
    $timestamp = strtotime($date);
    $dayOfYear = date('z', $timestamp) + 1;
    
    // Calculate sun position
    $declination = -23.45 * cos(deg2rad(360/365 * ($dayOfYear + 10)));
    
    // Bangladesh timezone offset (UTC+6)
    $timezone = 6;
    
    // Longitude correction (from standard meridian 90°E for Bangladesh)
    $longitudeCorrection = ($longitude - 90) * 4 / 60;
    
    // Equation of time (simplified)
    $B = 360/365 * ($dayOfYear - 81);
    $equationOfTime = 9.87 * sin(deg2rad(2*$B)) - 7.53 * cos(deg2rad($B)) - 1.5 * sin(deg2rad($B));
    $equationOfTime = $equationOfTime / 60; // Convert to hours
    
    // Solar noon
    $solarNoon = 12 - $longitudeCorrection - $equationOfTime;
    
    // Hour angle calculations
    $latRad = deg2rad($latitude);
    $declRad = deg2rad($declination);
    
    // Fajr (18° below horizon - Hanafi method for Bangladesh)
    $fajrAngle = -18;
    $cosHourAngleFajr = (sin(deg2rad($fajrAngle)) - sin($latRad) * sin($declRad)) / (cos($latRad) * cos($declRad));
    $hourAngleFajr = rad2deg(acos(max(-1, min(1, $cosHourAngleFajr)))) / 15;
    $fajr = $solarNoon - $hourAngleFajr;
    
    // Sunrise (0.833° below horizon for atmospheric refraction)
    $sunriseAngle = -0.833;
    $cosHourAngleSunrise = (sin(deg2rad($sunriseAngle)) - sin($latRad) * sin($declRad)) / (cos($latRad) * cos($declRad));
    $hourAngleSunrise = rad2deg(acos(max(-1, min(1, $cosHourAngleSunrise)))) / 15;
    $sunrise = $solarNoon - $hourAngleSunrise;
    
    // Dhuhr (solar noon + small adjustment)
    $dhuhr = $solarNoon + 0.0167; // Add 1 minute safety margin
    
    // Asr (Hanafi: shadow = 2x object + noon shadow)
    $noonShadowRatio = abs(tan($latRad - $declRad));
    $asrShadowRatio = 2 + $noonShadowRatio; // Hanafi method
    $asrAngle = rad2deg(atan(1 / $asrShadowRatio));
    $cosHourAngleAsr = (sin(deg2rad($asrAngle)) - sin($latRad) * sin($declRad)) / (cos($latRad) * cos($declRad));
    $hourAngleAsr = rad2deg(acos(max(-1, min(1, $cosHourAngleAsr)))) / 15;
    $asr = $solarNoon + $hourAngleAsr;
    
    // Maghrib (sunset)
    $maghrib = $solarNoon + $hourAngleSunrise;
    
    // Isha (18° below horizon)
    $ishaAngle = -18;
    $cosHourAngleIsha = (sin(deg2rad($ishaAngle)) - sin($latRad) * sin($declRad)) / (cos($latRad) * cos($declRad));
    $hourAngleIsha = rad2deg(acos(max(-1, min(1, $cosHourAngleIsha)))) / 15;
    $isha = $solarNoon + $hourAngleIsha;
    
    // Sehri end (10 minutes before Fajr)
    $sehriEnd = $fajr - (10/60);
    
    // Iftar (same as Maghrib)
    $iftar = $maghrib;
    
    // Convert to HH:MM format
    return [
        'fajr' => formatTime($fajr),
        'sunrise' => formatTime($sunrise),
        'dhuhr' => formatTime($dhuhr),
        'asr' => formatTime($asr),
        'maghrib' => formatTime($maghrib),
        'isha' => formatTime($isha),
        'sehri_end' => formatTime($sehriEnd),
        'iftar' => formatTime($iftar),
    ];
}

function formatTime($decimalHours) {
    // Ensure within 24 hour range
    while ($decimalHours < 0) $decimalHours += 24;
    while ($decimalHours >= 24) $decimalHours -= 24;
    
    $hours = floor($decimalHours);
    $minutes = round(($decimalHours - $hours) * 60);
    
    if ($minutes >= 60) {
        $minutes -= 60;
        $hours += 1;
    }
    if ($hours >= 24) {
        $hours -= 24;
    }
    
    return sprintf('%02d:%02d', $hours, $minutes);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'generate') {
        $locationId = $_POST['location_id'] ?? 'all';
        $startDate = $_POST['start_date'];
        $endDate = $_POST['end_date'];
        $ramadanStartDay = (int)($_POST['ramadan_start_day'] ?? 1);
        $overwrite = isset($_POST['overwrite']);
        $clearExisting = isset($_POST['clear_existing']);
        
        // Clear existing prayer times if requested
        if ($clearExisting) {
            if ($locationId === 'all') {
                // Delete for all locations, not just those with coordinates
                $clearStmt = $pdo->prepare("DELETE FROM prayer_times WHERE date BETWEEN ? AND ?");
                $clearStmt->execute([$startDate, $endDate]);
                $deletedCount = $clearStmt->rowCount();
            } else {
                $clearStmt = $pdo->prepare("DELETE FROM prayer_times WHERE location_id = ? AND date BETWEEN ? AND ?");
                $clearStmt->execute([$locationId, $startDate, $endDate]);
                $deletedCount = $clearStmt->rowCount();
            }
            $overwrite = true; // Force overwrite when clearing
            
            // Add feedback about deletion
            if ($deletedCount > 0) {
                $message = '<div style="background: #ffeaa7; color: #6c5ce7; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Deleted ' . $deletedCount . ' existing prayer time entries. Now generating new ones...</div>';
            }
        }
        
        // Get locations
        if ($locationId === 'all') {
            $locations = $pdo->query("SELECT * FROM locations WHERE is_active = 1 AND latitude IS NOT NULL AND longitude IS NOT NULL")->fetchAll();
        } else {
            $stmt = $pdo->prepare("SELECT * FROM locations WHERE id = ? AND latitude IS NOT NULL AND longitude IS NOT NULL");
            $stmt->execute([$locationId]);
            $locations = $stmt->fetchAll();
        }
        
        if (empty($locations)) {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">No locations found with coordinates. Please add latitude/longitude to locations first.</div>';
        } else {
            $currentDate = $startDate;
            $ramadanDay = $ramadanStartDay;
            
            // Prepare insert statement
            $insertStmt = $pdo->prepare("
                INSERT INTO prayer_times 
                (location_id, date, fajr, sunrise, dhuhr, asr, maghrib, isha, sehri_end, iftar, ramadan_day)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                fajr = VALUES(fajr),
                sunrise = VALUES(sunrise),
                dhuhr = VALUES(dhuhr),
                asr = VALUES(asr),
                maghrib = VALUES(maghrib),
                isha = VALUES(isha),
                sehri_end = VALUES(sehri_end),
                iftar = VALUES(iftar),
                ramadan_day = VALUES(ramadan_day)
            ");
            
            while (strtotime($currentDate) <= strtotime($endDate)) {
                foreach ($locations as $loc) {
                    // Check if entry exists and skip if not overwriting
                    if (!$overwrite) {
                        $checkStmt = $pdo->prepare("SELECT id FROM prayer_times WHERE location_id = ? AND date = ?");
                        $checkStmt->execute([$loc['id'], $currentDate]);
                        if ($checkStmt->fetch()) {
                            continue;
                        }
                    }
                    
                    // Calculate prayer times
                    $times = calculatePrayerTimes($loc['latitude'], $loc['longitude'], $currentDate);
                    
                    try {
                        $insertStmt->execute([
                            $loc['id'],
                            $currentDate,
                            $times['fajr'],
                            $times['sunrise'],
                            $times['dhuhr'],
                            $times['asr'],
                            $times['maghrib'],
                            $times['isha'],
                            $times['sehri_end'],
                            $times['iftar'],
                            $ramadanDay > 30 ? 0 : $ramadanDay,
                        ]);
                        $generated++;
                    } catch (Exception $e) {
                        // Log error but continue
                        error_log("Failed to insert prayer time: " . $e->getMessage());
                    }
                }
                
                $currentDate = date('Y-m-d', strtotime($currentDate . ' +1 day'));
                $ramadanDay++;
            }
            
            $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Successfully generated ' . $generated . ' prayer time entries!</div>';
        }
    } elseif ($_POST['action'] === 'delete_times') {
        $locationId = $_POST['location_id'] ?? 'all';
        $startDate = $_POST['start_date'];
        $endDate = $_POST['end_date'];
        
        // Delete prayer times
        if ($locationId === 'all') {
            $deleteStmt = $pdo->prepare("DELETE FROM prayer_times WHERE date BETWEEN ? AND ?");
            $deleteStmt->execute([$startDate, $endDate]);
            $deletedCount = $deleteStmt->rowCount();
        } else {
            $deleteStmt = $pdo->prepare("DELETE FROM prayer_times WHERE location_id = ? AND date BETWEEN ? AND ?");
            $deleteStmt->execute([$locationId, $startDate, $endDate]);
            $deletedCount = $deleteStmt->rowCount();
        }
        
        if ($deletedCount > 0) {
            $message = '<div style="background: #ffeaa7; color: #d63031; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Successfully deleted ' . $deletedCount . ' prayer time entries!</div>';
        } else {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">No prayer time entries found to delete in the selected date range.</div>';
        }
    } elseif ($_POST['action'] === 'upload_csv') {
        if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
            $csvFile = $_FILES['csv_file']['tmp_name'];
            $csvName = $_FILES['csv_file']['name'];
            $overwrite = isset($_POST['overwrite']);
            
            // Validate file extension
            $fileExtension = strtolower(pathinfo($csvName, PATHINFO_EXTENSION));
            if ($fileExtension !== 'csv') {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please upload a CSV file only.</div>';
            } else {
                $uploadedCount = 0;
                $errorCount = 0;
                $errors = [];
                
                // Prepare insert statement
                $insertStmt = $pdo->prepare("
                    INSERT INTO prayer_times 
                    (location_id, date, fajr, sunrise, dhuhr, asr, maghrib, isha, sehri_end, iftar, ramadan_day)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                    fajr = VALUES(fajr),
                    sunrise = VALUES(sunrise),
                    dhuhr = VALUES(dhuhr),
                    asr = VALUES(asr),
                    maghrib = VALUES(maghrib),
                    isha = VALUES(isha),
                    sehri_end = VALUES(sehri_end),
                    iftar = VALUES(iftar),
                    ramadan_day = VALUES(ramadan_day)
                ");
                
                // Open and read CSV file
                if (($handle = fopen($csvFile, 'r')) !== FALSE) {
                    $header = fgetcsv($handle); // Skip header row
                    
                    while (($row = fgetcsv($handle)) !== FALSE) {
                        try {
                            // Expected CSV format: location_id, date, fajr, sunrise, dhuhr, asr, maghrib, isha, sehri_end, iftar, ramadan_day
                            if (count($row) >= 11) {
                                $locationId = trim($row[0]);
                                $date = trim($row[1]);
                                $fajr = trim($row[2]);
                                $sunrise = trim($row[3]);
                                $dhuhr = trim($row[4]);
                                $asr = trim($row[5]);
                                $maghrib = trim($row[6]);
                                $isha = trim($row[7]);
                                $sehriEnd = trim($row[8]);
                                $iftar = trim($row[9]);
                                $ramadanDay = (int)trim($row[10]);
                                
                                // Validate required fields
                                if (empty($locationId) || empty($date) || empty($fajr) || empty($sunrise) || empty($dhuhr) || empty($asr) || empty($maghrib) || empty($isha)) {
                                    $errors[] = "Row " . ($uploadedCount + $errorCount + 2) . ": Missing required fields";
                                    $errorCount++;
                                    continue;
                                }
                                
                                // Validate date format
                                if (!DateTime::createFromFormat('Y-m-d', $date)) {
                                    $errors[] = "Row " . ($uploadedCount + $errorCount + 2) . ": Invalid date format (use Y-m-d)";
                                    $errorCount++;
                                    continue;
                                }
                                
                                // Auto-calculate Sehri End and Iftar if empty
                                if (empty($sehriEnd)) {
                                    $sehriEnd = date('H:i', strtotime($fajr . ' -10 minutes'));
                                }
                                if (empty($iftar)) {
                                    $iftar = $maghrib;
                                }
                                
                                $insertStmt->execute([
                                    $locationId,
                                    $date,
                                    $fajr,
                                    $sunrise,
                                    $dhuhr,
                                    $asr,
                                    $maghrib,
                                    $isha,
                                    $sehriEnd,
                                    $iftar,
                                    $ramadanDay > 30 ? 0 : $ramadanDay,
                                ]);
                                
                                $uploadedCount++;
                            } else {
                                $errors[] = "Row " . ($uploadedCount + $errorCount + 2) . ": Insufficient columns (expected 11)";
                                $errorCount++;
                            }
                        } catch (Exception $e) {
                            $errors[] = "Row " . ($uploadedCount + $errorCount + 2) . ": " . $e->getMessage();
                            $errorCount++;
                        }
                    }
                    fclose($handle);
                    
                    // Create success/error message
                    if ($uploadedCount > 0) {
                        $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">';
                        $message .= "Successfully uploaded $uploadedCount prayer time entries!";
                        if ($errorCount > 0) {
                            $message .= " ($errorCount rows had errors)";
                        }
                        $message .= '</div>';
                    } else {
                        $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">No valid entries found in CSV file.</div>';
                    }
                    
                    // Show errors if any
                    if (!empty($errors) && count($errors) <= 10) {
                        $message .= '<div style="background: #fff3cd; color: #856404; padding: 10px; border-radius: 6px; margin-bottom: 20px;">';
                        $message .= '<strong>Errors:</strong><ul>';
                        foreach ($errors as $error) {
                            $message .= '<li>' . htmlspecialchars($error) . '</li>';
                        }
                        $message .= '</ul></div>';
                    } elseif (!empty($errors)) {
                        $message .= '<div style="background: #fff3cd; color: #856404; padding: 10px; border-radius: 6px; margin-bottom: 20px;">';
                        $message .= '<strong>Errors:</strong> ' . count($errors) . ' errors occurred (first 10 shown)<ul>';
                        foreach (array_slice($errors, 0, 10) as $error) {
                            $message .= '<li>' . htmlspecialchars($error) . '</li>';
                        }
                        $message .= '</ul></div>';
                    }
                } else {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error opening CSV file.</div>';
                }
            }
        } else {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please select a CSV file to upload.</div>';
        }
    } elseif ($_POST['action'] === 'manual_generate') {
        $locationId = $_POST['location_id'] ?? '';
        $date = $_POST['date'] ?? '';
        $fajr = $_POST['fajr'] ?? '';
        $sunrise = $_POST['sunrise'] ?? '';
        $dhuhr = $_POST['dhuhr'] ?? '';
        $asr = $_POST['asr'] ?? '';
        $maghrib = $_POST['maghrib'] ?? '';
        $isha = $_POST['isha'] ?? '';
        $sehriEnd = $_POST['sehri_end'] ?? '';
        $iftar = $_POST['iftar'] ?? '';
        $ramadanDay = (int)($_POST['ramadan_day'] ?? 0);
        $overwrite = isset($_POST['overwrite']);
        
        // Validation
        if (empty($locationId) || empty($date) || empty($fajr) || empty($sunrise) || empty($dhuhr) || empty($asr) || empty($maghrib) || empty($isha)) {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please fill in all required fields.</div>';
        } else {
            // Auto-calculate Sehri End if empty (10 minutes before Fajr)
            if (empty($sehriEnd)) {
                $sehriEnd = date('H:i', strtotime($fajr . ' -10 minutes'));
            }
            
            // Auto-calculate Iftar if empty (same as Maghrib)
            if (empty($iftar)) {
                $iftar = $maghrib;
            }
            
            // Check if entry exists
            if (!$overwrite) {
                $checkStmt = $pdo->prepare("SELECT id FROM prayer_times WHERE location_id = ? AND date = ?");
                $checkStmt->execute([$locationId, $date]);
                if ($checkStmt->fetch()) {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Prayer times already exist for this location and date. Check "Overwrite existing entry" to update.</div>';
                }
            }
            
            if (empty($message)) {
                try {
                    $insertStmt = $pdo->prepare("
                        INSERT INTO prayer_times 
                        (location_id, date, fajr, sunrise, dhuhr, asr, maghrib, isha, sehri_end, iftar, ramadan_day)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                        ON DUPLICATE KEY UPDATE
                        fajr = VALUES(fajr),
                        sunrise = VALUES(sunrise),
                        dhuhr = VALUES(dhuhr),
                        asr = VALUES(asr),
                        maghrib = VALUES(maghrib),
                        isha = VALUES(isha),
                        sehri_end = VALUES(sehri_end),
                        iftar = VALUES(iftar),
                        ramadan_day = VALUES(ramadan_day)
                    ");
                    
                    $insertStmt->execute([
                        $locationId,
                        $date,
                        $fajr,
                        $sunrise,
                        $dhuhr,
                        $asr,
                        $maghrib,
                        $isha,
                        $sehriEnd,
                        $iftar,
                        $ramadanDay > 30 ? 0 : $ramadanDay,
                    ]);
                    
                    $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Successfully saved manual prayer times!</div>';
                } catch (Exception $e) {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error saving prayer times: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
            }
        }
    }
}

// Get locations for dropdown
$locations = $pdo->query("SELECT id, name, district, latitude, longitude FROM locations WHERE is_active = 1 ORDER BY district, name")->fetchAll();

// Get stats
$totalPrayerTimes = $pdo->query("SELECT COUNT(*) FROM prayer_times")->fetchColumn();
$locationsWithCoords = $pdo->query("SELECT COUNT(*) FROM locations WHERE latitude IS NOT NULL AND longitude IS NOT NULL AND is_active = 1")->fetchColumn();
?>
<?php include 'header.php'; ?>

<div class="container">
    <h2>Generate Prayer Times</h2>
    
    <?= $message ?>
    
    <div class="stats-row">
        <div class="stat-card">
            <h4>Total Prayer Times</h4>
            <p class="stat-number"><?= number_format($totalPrayerTimes) ?></p>
        </div>
        <div class="stat-card">
            <h4>Locations with Coordinates</h4>
            <p class="stat-number"><?= $locationsWithCoords ?></p>
        </div>
    </div>
    
    <div class="section">
        <h3>Bulk CSV Upload</h3>
        <p style="color: #666; margin-bottom: 20px;">
            Upload prayer times in bulk using a CSV file. This is the fastest way to import large amounts of data.
        </p>
        
        <div class="csv-format-info" style="background: #e3f2fd; padding: 15px; border-radius: 6px; margin-bottom: 20px;">
            <h4>CSV Format Required:</h4>
            <p><strong>Columns (11 total):</strong></p>
            <ol style="margin: 10px 0; padding-left: 20px;">
                <li><code>location_id</code> - Location ID number</li>
                <li><code>date</code> - Date in Y-m-d format (e.g., 2026-03-01)</li>
                <li><code>fajr</code> - Fajr time (HH:MM format)</li>
                <li><code>sunrise</code> - Sunrise time (HH:MM format)</li>
                <li><code>dhuhr</code> - Dhuhr time (HH:MM format)</li>
                <li><code>asr</code> - Asr time (HH:MM format)</li>
                <li><code>maghrib</code> - Maghrib time (HH:MM format)</li>
                <li><code>isha</code> - Isha time (HH:MM format)</li>
                <li><code>sehri_end</code> - Sehri end time (HH:MM, optional)</li>
                <li><code>iftar</code> - Iftar time (HH:MM, optional)</li>
                <li><code>ramadan_day</code> - Ramadan day number (0-30, 0 = Not Ramadan)</li>
            </ol>
            <p><strong>Example Row:</strong> <code>1,2026-03-01,04:30,05:45,12:15,15:30,18:00,19:15,04:20,18:00,1</code></p>
            <p><small>Note: If sehri_end or iftar are empty, they will be auto-calculated.</small></p>
        </div>
        
        <form method="POST" enctype="multipart/form-data" style="max-width: 600px;">
            <input type="hidden" name="action" value="upload_csv">
            
            <div class="form-group">
                <label>CSV File *</label>
                <input type="file" name="csv_file" accept=".csv" required style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;">
                <small style="color: #666; display: block; margin-top: 5px;">Select a CSV file with prayer times data</small>
            </div>
            
            <div class="form-group">
                <label>
                    <input type="checkbox" name="overwrite" checked> Overwrite existing entries
                </label>
                <small style="color: #666; display: block; margin-top: 5px;">Update existing prayer times if they already exist</small>
            </div>
            
            <button type="submit" class="btn btn-primary" onclick="return confirm('Upload CSV file and import prayer times?')">
                📁 Upload CSV File
            </button>
        </form>
        
        <div style="margin-top: 20px;">
            <a href="javascript:void(0)" onclick="downloadSampleCSV()" style="color: #2e7d32; text-decoration: none;">
                📥 Download Sample CSV Template
            </a>
        </div>
    </div>
    
    <div class="section">
        <h3>Delete Prayer Times</h3>
        <p style="color: #666; margin-bottom: 20px;">
            Delete existing prayer times for specific locations and date ranges. This action cannot be undone.
        </p>
        
        <form method="POST" style="max-width: 600px;">
            <input type="hidden" name="action" value="delete_times">
            
            <div class="form-group">
                <label>Location</label>
                <select name="location_id" style="width: 100%; padding: 8px;">
                    <option value="all">All Locations</option>
                    <?php foreach ($locations as $loc): ?>
                        <option value="<?= $loc['id'] ?>">
                            <?= htmlspecialchars($loc['name'] . ', ' . $loc['district']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Start Date *</label>
                    <input type="date" name="start_date" required value="<?= date('Y-m-d') ?>">
                </div>
                <div class="form-group">
                    <label>End Date *</label>
                    <input type="date" name="end_date" required value="<?= date('Y-m-d', strtotime('+30 days')) ?>">
                </div>
            </div>
            
            <button type="submit" class="btn btn-danger" onclick="return confirmDeletePrayerTimes(event)">
                🗑️ Delete Prayer Times
            </button>
        </form>
    </div>
    
    <div class="section">
        <h3>Auto-Generate Prayer Times</h3>
        <p style="color: #666; margin-bottom: 20px;">
            This will calculate prayer times based on geographic coordinates using Islamic astronomical calculations (Hanafi method).
        </p>
        
        <form method="POST" style="max-width: 600px;">
            <input type="hidden" name="action" value="generate">
            
            <div class="form-group">
                <label>Location</label>
                <select name="location_id" style="width: 100%; padding: 8px;">
                    <option value="all">All Locations (with coordinates)</option>
                    <?php foreach ($locations as $loc): ?>
                        <?php if ($loc['latitude'] && $loc['longitude']): ?>
                            <option value="<?= $loc['id'] ?>">
                                <?= htmlspecialchars($loc['name'] . ', ' . $loc['district']) ?>
                                (<?= round($loc['latitude'], 2) ?>°, <?= round($loc['longitude'], 2) ?>°)
                            </option>
                        <?php endif; ?>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Start Date *</label>
                    <input type="date" name="start_date" required value="<?= date('Y-m-d') ?>">
                </div>
                <div class="form-group">
                    <label>End Date *</label>
                    <input type="date" name="end_date" required value="<?= date('Y-m-d', strtotime('+30 days')) ?>">
                </div>
            </div>
            
            <div class="form-group">
                <label>Ramadan Day (starting from start date)</label>
                <input type="number" name="ramadan_start_day" min="0" max="30" value="1" placeholder="0 = Not Ramadan">
                <small style="color: #666; display: block; margin-top: 5px;">Enter 0 if not during Ramadan period</small>
            </div>
            
            <div class="form-group">
                <label>
                    <input type="checkbox" name="overwrite"> Overwrite existing entries
                </label>
            </div>
            
            <div class="form-group">
                <label>
                    <input type="checkbox" name="clear_existing" onclick="toggleOverwrite(this)"> Clear existing prayer times for selected date range first
                </label>
                <small style="color: #666; display: block; margin-top: 5px;">This will delete all existing prayer times before generating new ones. Use with caution!</small>
            </div>
            
            <button type="submit" class="btn" onclick="return confirmClearAndGenerate(event)">
                Generate Prayer Times
            </button>
        </form>
    </div>
    
    <script>
    function downloadSampleCSV() {
        const csvContent = "location_id,date,fajr,sunrise,dhuhr,asr,maghrib,isha,sehri_end,iftar,ramadan_day\n" +
                           "1,2026-03-01,04:30,05:45,12:15,15:30,18:00,19:15,04:20,18:00,1\n" +
                           "1,2026-03-02,04:29,05:44,12:15,15:31,18:01,19:16,04:19,18:01,2\n" +
                           "2,2026-03-01,04:32,05:47,12:17,15:32,18:02,19:17,04:22,18:02,1";
        
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'prayer_times_sample.csv';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
    }
    
    function toggleOverwrite(checkbox) {
        const overwriteCheckbox = document.querySelector('input[name="overwrite"]');
        if (checkbox.checked) {
            overwriteCheckbox.checked = true;
            overwriteCheckbox.disabled = true;
        } else {
            overwriteCheckbox.disabled = false;
        }
    }
    
    function confirmClearAndGenerate(event) {
        const clearCheckbox = document.querySelector('input[name="clear_existing"]');
        const locationSelect = document.querySelector('select[name="location_id"]');
        const startDate = document.querySelector('input[name="start_date"]').value;
        const endDate = document.querySelector('input[name="end_date"]').value;
        const locationText = locationSelect.options[locationSelect.selectedIndex].text;
        
        let message = 'Generate prayer times';
        
        if (clearCheckbox.checked) {
            message = '⚠️ WARNING: This will DELETE all existing prayer times for:\\n\\n';
            message += 'Location: ' + locationText + '\\n';
            message += 'Date Range: ' + startDate + ' to ' + endDate + '\\n\\n';
            message += 'This action cannot be undone!\\n\\n';
            message += 'Are you sure you want to continue?';
        } else {
            message += ' for the selected date range?';
        }
        
        return confirm(message);
    }
    
    function confirmDeletePrayerTimes(event) {
        const locationSelect = document.querySelector('select[name="location_id"]');
        const startDate = document.querySelector('input[name="start_date"]').value;
        const endDate = document.querySelector('input[name="end_date"]').value;
        const locationText = locationSelect.options[locationSelect.selectedIndex].text;
        
        const message = '⚠️ DANGER: This will PERMANENTLY DELETE all prayer times for:\\n\\n';
        message += 'Location: ' + locationText + '\\n';
        message += 'Date Range: ' + startDate + ' to ' + endDate + '\\n\\n';
        message += 'This action CANNOT be undone!\\n\\n';
        message += 'Are you absolutely sure you want to delete?';
        
        return confirm(message);
    }
    </script>
    
    <div class="section">
        <h3>Manual-Generate Prayer Times</h3>
        <p style="color: #666; margin-bottom: 20px;">
            Enter prayer times manually for a specific location and date. This is useful when you need to override calculated times or add times for locations without coordinates.
        </p>
        
        <form method="POST" style="max-width: 600px;" id="manualForm">
            <input type="hidden" name="action" value="manual_generate">
            
            <div class="form-group">
                <label>Location *</label>
                <select name="location_id" style="width: 100%; padding: 8px;" required>
                    <option value="">Select Location</option>
                    <?php foreach ($locations as $loc): ?>
                        <option value="<?= $loc['id'] ?>">
                            <?= htmlspecialchars($loc['name'] . ', ' . $loc['district']) ?>
                            <?php if ($loc['latitude'] && $loc['longitude']): ?>
                                (<?= round($loc['latitude'], 2) ?>°, <?= round($loc['longitude'], 2) ?>°)
                            <?php endif; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label>Date *</label>
                <input type="date" name="date" required value="<?= date('Y-m-d') ?>">
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Fajr *</label>
                    <input type="time" name="fajr" required>
                </div>
                <div class="form-group">
                    <label>Sunrise *</label>
                    <input type="time" name="sunrise" required>
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Dhuhr *</label>
                    <input type="time" name="dhuhr" required>
                </div>
                <div class="form-group">
                    <label>Asr *</label>
                    <input type="time" name="asr" required>
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Maghrib *</label>
                    <input type="time" name="maghrib" required>
                </div>
                <div class="form-group">
                    <label>Isha *</label>
                    <input type="time" name="isha" required>
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Sehri End</label>
                    <input type="time" name="sehri_end">
                    <small style="color: #666; display: block; margin-top: 5px;">Leave empty to auto-calculate (10 min before Fajr)</small>
                </div>
                <div class="form-group">
                    <label>Iftar</label>
                    <input type="time" name="iftar">
                    <small style="color: #666; display: block; margin-top: 5px;">Leave empty to use Maghrib time</small>
                </div>
            </div>
            
            <div class="form-group">
                <label>Ramadan Day</label>
                <input type="number" name="ramadan_day" min="0" max="30" value="0" placeholder="0 = Not Ramadan">
                <small style="color: #666; display: block; margin-top: 5px;">Enter 0 if not during Ramadan period</small>
            </div>
            
            <div class="form-group">
                <label>
                    <input type="checkbox" name="overwrite"> Overwrite existing entry
                </label>
            </div>
            
            <button type="submit" class="btn" onclick="return confirm('Save manual prayer times?')">
                Save Manual Prayer Times
            </button>
        </form>
    </div>
    
    <div class="section">
        <h3>Quick Generate Presets</h3>
        <div class="preset-buttons">
            <form method="POST" style="display: inline;">
                <input type="hidden" name="action" value="generate">
                <input type="hidden" name="location_id" value="all">
                <input type="hidden" name="start_date" value="<?= date('Y-m-d') ?>">
                <input type="hidden" name="end_date" value="<?= date('Y-m-d') ?>">
                <input type="hidden" name="ramadan_start_day" value="0">
                <input type="hidden" name="overwrite" value="1">
                <button type="submit" class="btn btn-secondary">Today Only</button>
            </form>
            
            <form method="POST" style="display: inline;">
                <input type="hidden" name="action" value="generate">
                <input type="hidden" name="location_id" value="all">
                <input type="hidden" name="start_date" value="<?= date('Y-m-d') ?>">
                <input type="hidden" name="end_date" value="<?= date('Y-m-d', strtotime('+7 days')) ?>">
                <input type="hidden" name="ramadan_start_day" value="0">
                <input type="hidden" name="overwrite" value="1">
                <button type="submit" class="btn btn-secondary">Next 7 Days</button>
            </form>
            
            <form method="POST" style="display: inline;">
                <input type="hidden" name="action" value="generate">
                <input type="hidden" name="location_id" value="all">
                <input type="hidden" name="start_date" value="2026-02-18">
                <input type="hidden" name="end_date" value="2026-03-19">
                <input type="hidden" name="ramadan_start_day" value="1">
                <input type="hidden" name="overwrite" value="1">
                <button type="submit" class="btn btn-primary">Ramadan 2026 (Feb 18 - Mar 19)</button>
            </form>
        </div>
    </div>
    
    <div class="section">
        <h3>Locations Missing Coordinates</h3>
        <?php 
        $missingCoords = $pdo->query("SELECT id, name, district FROM locations WHERE (latitude IS NULL OR longitude IS NULL) AND is_active = 1")->fetchAll();
        if (empty($missingCoords)):
        ?>
            <p style="color: #060;">All active locations have coordinates configured.</p>
        <?php else: ?>
            <p style="color: #c60;">The following locations need coordinates to generate prayer times:</p>
            <ul>
                <?php foreach ($missingCoords as $loc): ?>
                    <li><?= htmlspecialchars($loc['name'] . ', ' . $loc['district']) ?> - <a href="locations.php">Edit</a></li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>
    </div>
</div>

<style>
    .stats-row { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px; }
    .stat-card { background: #f8f9fa; padding: 20px; border-radius: 8px; text-align: center; }
    .stat-card h4 { margin: 0 0 10px 0; color: #666; font-size: 14px; }
    .stat-number { font-size: 32px; font-weight: bold; color: #2e7d32; margin: 0; }
    .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
    .form-group { margin-bottom: 15px; }
    .form-group label { display: block; margin-bottom: 5px; font-weight: 500; }
    .form-group input, .form-group select { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
    .preset-buttons { display: flex; gap: 10px; flex-wrap: wrap; }
    .btn-secondary { background: #6c757d !important; }
    .btn-primary { background: #2e7d32 !important; }
    .btn-danger { background: #d32f2f !important; color: white !important; }
    .btn-danger:hover { background: #b71c1c !important; }
</style>
</body>
</html>
