<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';
$editItem = null;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'add') {
        $stmt = $pdo->prepare("
            INSERT INTO islamic_content 
            (type, title, content, arabic_text, transliteration, translation, display_order, is_active)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        try {
            $stmt->execute([
                $_POST['type'],
                $_POST['title'],
                $_POST['content'],
                $_POST['arabic_text'] ?? null,
                $_POST['transliteration'] ?? null,
                $_POST['translation'] ?? null,
                $_POST['display_order'] ?? 0,
                isset($_POST['is_active']) ? 1 : 0,
            ]);
            $message = '<div class="alert alert-success">Content added successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    elseif (isset($_POST['action']) && $_POST['action'] === 'edit') {
        $stmt = $pdo->prepare("
            UPDATE islamic_content 
            SET type = ?, title = ?, content = ?, arabic_text = ?, transliteration = ?, 
                translation = ?, display_order = ?, is_active = ?
            WHERE id = ?
        ");
        
        try {
            $stmt->execute([
                $_POST['type'],
                $_POST['title'],
                $_POST['content'],
                $_POST['arabic_text'] ?? null,
                $_POST['transliteration'] ?? null,
                $_POST['translation'] ?? null,
                $_POST['display_order'] ?? 0,
                isset($_POST['is_active']) ? 1 : 0,
                $_POST['id']
            ]);
            $message = '<div class="alert alert-success">Content updated successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    elseif (isset($_POST['action']) && $_POST['action'] === 'delete') {
        $stmt = $pdo->prepare("DELETE FROM islamic_content WHERE id = ?");
        try {
            $stmt->execute([$_POST['id']]);
            $message = '<div class="alert alert-success">Content deleted successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// Handle edit request
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare("SELECT * FROM islamic_content WHERE id = ?");
    $stmt->execute([$_GET['edit']]);
    $editItem = $stmt->fetch();
}

// Get content
$content = $pdo->query("SELECT * FROM islamic_content ORDER BY type, display_order, id")->fetchAll();
?>
<?php include 'header.php'; ?>

<div class="container">
    <h2>Manage Islamic Content</h2>
    
    <?= $message ?>
    
    <div class="section">
        <h3><?= $editItem ? 'Edit Content' : 'Add New Content' ?></h3>
        <?php if ($editItem): ?>
            <a href="islamic-content.php" class="btn btn-secondary" style="margin-bottom: 15px; display: inline-block;">← Cancel Edit</a>
        <?php endif; ?>
        <form method="POST" style="max-width: 800px;">
            <input type="hidden" name="action" value="<?= $editItem ? 'edit' : 'add' ?>">
            <?php if ($editItem): ?>
                <input type="hidden" name="id" value="<?= $editItem['id'] ?>">
            <?php endif; ?>
            
            <div class="form-group">
                <label>Type *</label>
                <select name="type" required>
                    <option value="niyat" <?= ($editItem && $editItem['type'] == 'niyat') ? 'selected' : '' ?>>Roza Niyat</option>
                    <option value="sehri_dua" <?= ($editItem && $editItem['type'] == 'sehri_dua') ? 'selected' : '' ?>>Sehri Dua</option>
                    <option value="iftar_dua" <?= ($editItem && $editItem['type'] == 'iftar_dua') ? 'selected' : '' ?>>Iftar Dua</option>
                    <option value="masala" <?= ($editItem && $editItem['type'] == 'masala') ? 'selected' : '' ?>>Masala</option>
                </select>
            </div>
            <div class="form-group">
                <label>Title *</label>
                <input type="text" name="title" value="<?= $editItem ? htmlspecialchars($editItem['title']) : '' ?>" required>
            </div>
            <div class="form-group">
                <label>Content *</label>
                <textarea name="content" rows="5" required><?= $editItem ? htmlspecialchars($editItem['content']) : '' ?></textarea>
            </div>
            <div class="form-group">
                <label>Arabic Text</label>
                <textarea name="arabic_text" rows="3"><?= $editItem ? htmlspecialchars($editItem['arabic_text']) : '' ?></textarea>
            </div>
            <div class="form-group">
                <label>Transliteration</label>
                <textarea name="transliteration" rows="2"><?= $editItem ? htmlspecialchars($editItem['transliteration']) : '' ?></textarea>
            </div>
            <div class="form-group">
                <label>Translation</label>
                <textarea name="translation" rows="3"><?= $editItem ? htmlspecialchars($editItem['translation']) : '' ?></textarea>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Display Order</label>
                    <input type="number" name="display_order" value="<?= $editItem ? $editItem['display_order'] : '0' ?>" min="0">
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" name="is_active" <?= (!$editItem || $editItem['is_active']) ? 'checked' : '' ?>> Active
                    </label>
                </div>
            </div>
            <button type="submit" class="btn"><?= $editItem ? 'Update Content' : 'Add Content' ?></button>
        </form>
    </div>
    
    <div class="section">
        <h3>All Content (<?= count($content) ?>)</h3>
        <div style="overflow-x: auto;">
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Type</th>
                        <th>Title</th>
                        <th>Content Preview</th>
                        <th>Order</th>
                        <th>Status</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($content as $item): ?>
                        <tr>
                            <td><?= $item['id'] ?></td>
                            <td>
                                <span class="badge badge-<?= $item['type'] ?>">
                                    <?= htmlspecialchars($item['type']) ?>
                                </span>
                            </td>
                            <td><?= htmlspecialchars($item['title']) ?></td>
                            <td style="max-width: 300px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                <?= htmlspecialchars(substr($item['content'], 0, 100)) ?><?= strlen($item['content']) > 100 ? '...' : '' ?>
                            </td>
                            <td><?= $item['display_order'] ?></td>
                            <td>
                                <span class="badge <?= $item['is_active'] ? 'badge-active' : 'badge-inactive' ?>">
                                    <?= $item['is_active'] ? 'Active' : 'Inactive' ?>
                                </span>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <a href="?edit=<?= $item['id'] ?>" class="btn btn-small btn-primary">Edit</a>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this content?');">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="id" value="<?= $item['id'] ?>">
                                        <button type="submit" class="btn btn-small btn-danger">Delete</button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<style>
    .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
    .form-group { margin-bottom: 15px; }
    .form-group label { display: block; margin-bottom: 5px; font-weight: 500; }
    .form-group input, .form-group select, .form-group textarea { 
        width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; 
    }
    .form-group textarea { font-family: inherit; }
    
    .alert { padding: 12px 20px; border-radius: 6px; margin-bottom: 20px; }
    .alert-success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
    .alert-error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
    
    .badge { 
        display: inline-block; 
        padding: 4px 8px; 
        border-radius: 4px; 
        font-size: 12px; 
        font-weight: 500; 
    }
    .badge-niyat { background: #e3f2fd; color: #1976d2; }
    .badge-sehri_dua { background: #fff3e0; color: #f57c00; }
    .badge-iftar_dua { background: #fce4ec; color: #c2185b; }
    .badge-masala { background: #f3e5f5; color: #7b1fa2; }
    .badge-active { background: #d4edda; color: #155724; }
    .badge-inactive { background: #f8d7da; color: #721c24; }
    
    .action-buttons { display: flex; gap: 8px; flex-wrap: wrap; }
    .btn-small { padding: 6px 12px; font-size: 13px; }
    .btn-primary { background: #007bff; color: white; }
    .btn-primary:hover { background: #0056b3; }
    .btn-danger { background: #dc3545; color: white; }
    .btn-danger:hover { background: #c82333; }
    .btn-secondary { background: #6c757d; color: white; }
    .btn-secondary:hover { background: #5a6268; }
    
    table { font-size: 14px; }
    table td { vertical-align: middle; }
</style>
</body>
</html>
