<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';

// Restore flash message after redirect (POST-Redirect-GET)
if (isset($_SESSION['locations_message'])) {
    $message = $_SESSION['locations_message'];
    unset($_SESSION['locations_message']);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'add') {
            $stmt = $pdo->prepare("
                INSERT INTO locations (name, district, latitude, longitude, is_active)
                VALUES (?, ?, ?, ?, ?)
            ");
            
            try {
                $stmt->execute([
                    $_POST['name'],
                    $_POST['district'],
                    !empty($_POST['latitude']) ? $_POST['latitude'] : null,
                    !empty($_POST['longitude']) ? $_POST['longitude'] : null,
                    isset($_POST['is_active']) ? 1 : 0,
                ]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Location added successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['locations_message'] = $message;
            header('Location: locations.php');
            exit;
        } elseif ($_POST['action'] === 'update') {
            $stmt = $pdo->prepare("
                UPDATE locations 
                SET name = ?, district = ?, latitude = ?, longitude = ?, is_active = ?
                WHERE id = ?
            ");
            
            try {
                $stmt->execute([
                    $_POST['name'],
                    $_POST['district'],
                    !empty($_POST['latitude']) ? $_POST['latitude'] : null,
                    !empty($_POST['longitude']) ? $_POST['longitude'] : null,
                    isset($_POST['is_active']) ? 1 : 0,
                    $_POST['id'],
                ]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Location updated successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['locations_message'] = $message;
            header('Location: locations.php');
            exit;
        } elseif ($_POST['action'] === 'delete') {
            $stmt = $pdo->prepare("DELETE FROM locations WHERE id = ?");
            try {
                $stmt->execute([$_POST['id']]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Location deleted successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['locations_message'] = $message;
            header('Location: locations.php');
            exit;
        } elseif ($_POST['action'] === 'bulk_delete') {
            if (isset($_POST['selected_ids']) && is_array($_POST['selected_ids']) && !empty($_POST['selected_ids'])) {
                $ids = array_map('intval', $_POST['selected_ids']);
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $stmt = $pdo->prepare("DELETE FROM locations WHERE id IN ($placeholders)");
                
                try {
                    $stmt->execute($ids);
                    $deletedCount = $stmt->rowCount();
                    $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Successfully deleted ' . $deletedCount . ' location(s)!</div>';
                } catch (Exception $e) {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
            } else {
                $message = '<div style="background: #ffd; color: #660; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please select at least one location to delete.</div>';
            }
            $_SESSION['locations_message'] = $message;
            header('Location: locations.php');
            exit;
        } elseif ($_POST['action'] === 'upload_csv') {
            if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
                $csvFile = $_FILES['csv_file']['tmp_name'];
                $csvName = $_FILES['csv_file']['name'];
                
                // Validate file extension
                $fileExtension = strtolower(pathinfo($csvName, PATHINFO_EXTENSION));
                if ($fileExtension !== 'csv') {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please upload a CSV file only.</div>';
                } else {
                    $uploadedCount = 0;
                    $errorCount = 0;
                    $errors = [];
                    $duplicateCount = 0;
                    
                    // Get the next auto-increment ID
                    $nextIdStmt = $pdo->query("SHOW TABLE STATUS LIKE 'locations'");
                    $tableStatus = $nextIdStmt->fetch();
                    $nextAutoId = $tableStatus['Auto_increment'];
                    
                    // Open and read CSV file
                    if (($handle = fopen($csvFile, 'r')) !== FALSE) {
                        $header = fgetcsv($handle, 1000, ','); // Skip header row
                        
                        $rowNumber = 2; // Start from row 2 (after header)
                        
                        while (($data = fgetcsv($handle, 1000, ',')) !== FALSE) {
                            try {
                                // Skip empty rows
                                if (empty(array_filter($data))) {
                                    $rowNumber++;
                                    continue;
                                }
                                
                                // Assign columns (expecting: name, district, latitude, longitude, is_active)
                                $name = trim($data[0] ?? '');
                                $district = trim($data[1] ?? '');
                                $latitude = !empty(trim($data[2] ?? '')) ? trim($data[2]) : null;
                                $longitude = !empty(trim($data[3] ?? '')) ? trim($data[3]) : null;
                                $isActive = !empty(trim($data[4] ?? '')) ? (strtolower(trim($data[4])) === 'active' || strtolower(trim($data[4])) === '1' || strtolower(trim($data[4])) === 'yes' || strtolower(trim($data[4])) === 'true') : 1;
                                
                                // Validate required fields
                                if (empty($name) || empty($district)) {
                                    $errors[] = "Row $rowNumber: Name and District are required";
                                    $errorCount++;
                                    $rowNumber++;
                                    continue;
                                }
                                
                                // Validate coordinates if provided
                                if ($latitude !== null && !is_numeric($latitude)) {
                                    $errors[] = "Row $rowNumber: Invalid latitude format";
                                    $errorCount++;
                                    $rowNumber++;
                                    continue;
                                }
                                
                                if ($longitude !== null && !is_numeric($longitude)) {
                                    $errors[] = "Row $rowNumber: Invalid longitude format";
                                    $errorCount++;
                                    $rowNumber++;
                                    continue;
                                }
                                
                                // Check for duplicate location (same name and district)
                                $duplicateCheck = $pdo->prepare("SELECT id FROM locations WHERE name = ? AND district = ?");
                                $duplicateCheck->execute([$name, $district]);
                                if ($duplicateCheck->fetch()) {
                                    $duplicateCount++;
                                    $rowNumber++;
                                    continue;
                                }
                                
                                // Insert location with automatic ID
                                $stmt = $pdo->prepare("
                                    INSERT INTO locations (id, name, district, latitude, longitude, is_active)
                                    VALUES (?, ?, ?, ?, ?, ?)
                                ");
                                
                                $stmt->execute([
                                    $nextAutoId, // Use the auto-increment ID
                                    $name,
                                    $district,
                                    $latitude,
                                    $longitude,
                                    $isActive ? 1 : 0
                                ]);
                                
                                $uploadedCount++;
                                $nextAutoId++; // Increment for next record
                                
                            } catch (Exception $e) {
                                $errors[] = "Row $rowNumber: " . $e->getMessage();
                                $errorCount++;
                            }
                            $rowNumber++;
                        }
                        fclose($handle);
                        
                        // Build success message
                        $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">';
                        $message .= "<strong>CSV Upload Complete!</strong><br>";
                        $message .= "✅ Successfully uploaded: <strong>$uploadedCount</strong> locations<br>";
                        
                        if ($duplicateCount > 0) {
                            $message .= "⚠️ Skipped duplicates: <strong>$duplicateCount</strong> locations<br>";
                        }
                        
                        if ($errorCount > 0) {
                            $message .= "❌ Errors: <strong>$errorCount</strong> rows<br>";
                            $message .= '<div style="margin-top: 10px; font-size: 12px; max-height: 100px; overflow-y: auto;">';
                            $message .= '<strong>Error Details:</strong><br>';
                            $message .= implode('<br>', array_slice($errors, 0, 10)); // Show first 10 errors
                            if (count($errors) > 10) {
                                $message .= '<br>... and ' . (count($errors) - 10) . ' more errors';
                            }
                            $message .= '</div>';
                        }
                        
                        $message .= '</div>';
                        
                    } else {
                        $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error opening CSV file.</div>';
                    }
                }
            } else {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please select a CSV file to upload.</div>';
            }
            $_SESSION['locations_message'] = $message;
            header('Location: locations.php');
            exit;
        }
    }
}

// Get locations
$locations = $pdo->query("SELECT * FROM locations ORDER BY district, name")->fetchAll();
?>
<?php include 'header.php'; ?>

<div class="container">
    <h2>Manage Locations</h2>
    
    <?= $message ?>
    
    <div class="section">
        <h3>Add New Location</h3>
        <form method="POST" style="max-width: 600px;">
            <input type="hidden" name="action" value="add">
            <div class="form-group">
                <label>Name *</label>
                <input type="text" name="name" required>
            </div>
            <div class="form-group">
                <label>District *</label>
                <input type="text" name="district" required>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Latitude</label>
                    <input type="number" step="any" name="latitude">
                </div>
                <div class="form-group">
                    <label>Longitude</label>
                    <input type="number" step="any" name="longitude">
                </div>
            </div>
            <div class="form-group">
                <label>
                    <input type="checkbox" name="is_active" checked> Active
                </label>
            </div>
            <button type="submit" class="btn">Add Location</button>
        </form>
    </div>
    
    <div class="section">
        <h3>Bulk CSV Upload</h3>
        <p style="color: #666; margin-bottom: 20px;">
            Upload locations in bulk using a CSV file. Location IDs will be generated automatically.
        </p>
        
        <div class="csv-format-info" style="background: #e3f2fd; padding: 15px; border-radius: 6px; margin-bottom: 20px;">
            <h4>CSV Format Required:</h4>
            <p><strong>Columns (5 total):</strong></p>
            <ol style="margin: 10px 0; padding-left: 20px;">
                <li><code>name</code> - Location name (required)</li>
                <li><code>district</code> - District name (required)</li>
                <li><code>latitude</code> - Latitude coordinate (optional)</li>
                <li><code>longitude</code> - Longitude coordinate (optional)</li>
                <li><code>is_active</code> - Status: active/1/yes/true or inactive/0/no/false (optional, defaults to active)</li>
            </ol>
            <p><strong>Example Row:</strong> <code>"New City","New District",23.8103,90.4125,"active"</code></p>
            <p><small>Note: Location IDs will be generated automatically. Duplicate locations (same name + district) will be skipped.</small></p>
        </div>
        
        <form method="POST" enctype="multipart/form-data" style="max-width: 600px;">
            <input type="hidden" name="action" value="upload_csv">
            
            <div class="form-group">
                <label>CSV File *</label>
                <input type="file" name="csv_file" accept=".csv" required style="width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px;">
                <small style="color: #666; display: block; margin-top: 5px;">Select a CSV file with location data</small>
            </div>
            
            <button type="submit" class="btn btn-primary" onclick="return confirm('Upload CSV file and import locations? Location IDs will be generated automatically.')">
                📁 Upload CSV File
            </button>
        </form>
        
        <div style="margin-top: 20px;">
            <a href="javascript:void(0)" onclick="downloadSampleCSV()" style="color: #2e7d32; text-decoration: none;">
                📥 Download Sample CSV Template
            </a>
        </div>
    </div>
    
    <div class="section">
        <h3>All Locations</h3>
        <?php if (empty($locations)): ?>
            <p style="color: #666; padding: 20px; text-align: center;">No locations found. Add locations using the form above.</p>
        <?php else: ?>
            <form method="POST" id="singleDeleteForm" style="display: none;">
                <input type="hidden" name="action" value="delete">
                <input type="hidden" name="id" id="singleDeleteId">
            </form>

            <form method="POST" id="bulkDeleteForm" onsubmit="return confirmBulkDelete();">
                <input type="hidden" name="action" value="bulk_delete">
                <div style="margin-bottom: 15px;">
                    <button type="submit" id="bulkDeleteBtn" class="btn btn-danger" disabled style="background: #dc3545; color: white; padding: 10px 20px; border: none; border-radius: 6px; cursor: not-allowed; font-weight: 600; opacity: 0.6;">
                        Delete Selected (<span id="selectedCount">0</span>)
                    </button>
                </div>

                <div style="overflow-x: auto;">
                    <table>
                        <thead>
                            <tr>
                                <th style="width: 50px;">
                                    <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                                </th>
                                <th>ID</th>
                                <th>Name</th>
                                <th>District</th>
                                <th>Coordinates</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($locations as $loc): ?>
                                <tr>
                                    <td style="text-align: center;">
                                        <input type="checkbox" name="selected_ids[]" value="<?= $loc['id'] ?>" class="row-checkbox" onchange="updateBulkDeleteButton()">
                                    </td>
                                    <td><?= htmlspecialchars($loc['id']) ?></td>
                                    <td><?= htmlspecialchars($loc['name']) ?></td>
                                    <td><?= htmlspecialchars($loc['district']) ?></td>
                                    <td>
                                        <?php if (!empty($loc['latitude']) && !empty($loc['longitude'])): ?>
                                            <?= htmlspecialchars($loc['latitude'] . ', ' . $loc['longitude']) ?>
                                        <?php else: ?>
                                            -
                                        <?php endif; ?>
                                    </td>
                                    <td><?= $loc['is_active'] ? 'Active' : 'Inactive' ?></td>
                                    <td>
                                        <button type="button" onclick="editEntry(<?= htmlspecialchars(json_encode($loc)) ?>)" class="btn-small" style="margin-right: 5px;">Edit</button>
                                        <button type="button" class="btn-small btn-danger" onclick="deleteEntry(<?= (int)$loc['id'] ?>)">Delete</button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </form>
        <?php endif; ?>
    </div>
</div>

<!-- Edit Modal -->
<div id="editModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000; align-items: center; justify-content: center;">
    <div style="background: white; padding: 30px; border-radius: 12px; max-width: 500px; width: 90%; max-height: 90vh; overflow-y: auto;">
        <h3 style="margin-bottom: 20px;">Edit Location</h3>
        <form method="POST" id="editForm">
            <input type="hidden" name="action" value="update">
            <input type="hidden" name="id" id="edit_id">
            <div class="form-group">
                <label>Name *</label>
                <input type="text" name="name" id="edit_name" required>
            </div>
            <div class="form-group">
                <label>District *</label>
                <input type="text" name="district" id="edit_district" required>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Latitude</label>
                    <input type="number" step="any" name="latitude" id="edit_latitude">
                </div>
                <div class="form-group">
                    <label>Longitude</label>
                    <input type="number" step="any" name="longitude" id="edit_longitude">
                </div>
            </div>
            <div class="form-group">
                <label>
                    <input type="checkbox" name="is_active" id="edit_is_active"> Active
                </label>
            </div>
            <div style="margin-top: 20px; display: flex; gap: 10px;">
                <button type="submit" class="btn">Update</button>
                <button type="button" onclick="closeEditModal()" class="btn" style="background: #666;">Cancel</button>
            </div>
        </form>
    </div>
</div>

<style>
    .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
    .form-group { margin-bottom: 15px; }
    .form-group label { display: block; margin-bottom: 5px; font-weight: 500; }
    .form-group input, .form-group select { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
    .btn-small { padding: 6px 12px; font-size: 12px; }
    .btn-danger { background: #c33; color: white; }
    .btn-danger:hover { background: #a22; }
    .btn-danger:disabled { background: #dc3545; opacity: 0.6; cursor: not-allowed; }
    .btn-danger:not(:disabled):hover { background: #c82333; opacity: 1; }
    .btn-primary { background: #007bff; color: white; }
    .btn-primary:hover { background: #0056b3; }
    .csv-format-info { border-left: 4px solid #2196F3; }
    .csv-format-info h4 { margin: 0 0 10px 0; color: #1976D2; }
    .csv-format-info code { background: #f5f5f5; padding: 2px 4px; border-radius: 3px; font-family: monospace; }
</style>

<script>
function deleteEntry(id) {
    if (!confirm('Are you sure you want to delete this location? Related prayer times may be affected.')) return;
    var f = document.getElementById('singleDeleteForm');
    if (f) {
        document.getElementById('singleDeleteId').value = id;
        f.submit();
    }
}

function downloadSampleCSV() {
    const csvContent = "name,district,latitude,longitude,is_active\n" +
                       "\"Dhaka\",\"Dhaka\",23.8103,90.4125,\"active\"\n" +
                       "\"Chittagong\",\"Chittagong\",22.3569,91.7832,\"active\"\n" +
                       "\"Sylhet\",\"Sylhet\",24.8949,91.8687,\"active\"\n" +
                       "\"Rajshahi\",\"Rajshahi\",24.3745,88.6042,\"active\"\n" +
                       "\"Barisal\",\"Barisal\",22.7010,90.3535,\"active\"";
    
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'locations_sample.csv';
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    window.URL.revokeObjectURL(url);
}

function editEntry(entry) {
    document.getElementById('edit_id').value = entry.id;
    document.getElementById('edit_name').value = entry.name || '';
    document.getElementById('edit_district').value = entry.district || '';
    document.getElementById('edit_latitude').value = entry.latitude ?? '';
    document.getElementById('edit_longitude').value = entry.longitude ?? '';
    document.getElementById('edit_is_active').checked = entry.is_active == 1;
    document.getElementById('editModal').style.display = 'flex';
}

function closeEditModal() {
    document.getElementById('editModal').style.display = 'none';
}

document.getElementById('editModal').addEventListener('click', function(e) {
    if (e.target === this) closeEditModal();
});

function toggleSelectAll(selectAllCheckbox) {
    document.querySelectorAll('.row-checkbox').forEach(function(cb) { cb.checked = selectAllCheckbox.checked; });
    updateBulkDeleteButton();
}

function updateBulkDeleteButton() {
    var checked = document.querySelectorAll('.row-checkbox:checked');
    var count = checked.length;
    var btn = document.getElementById('bulkDeleteBtn');
    var span = document.getElementById('selectedCount');
    if (span) span.textContent = count;
    if (btn) btn.disabled = count === 0;
    var selectAll = document.getElementById('selectAll');
    if (selectAll) {
        var all = document.querySelectorAll('.row-checkbox');
        var n = document.querySelectorAll('.row-checkbox:checked').length;
        selectAll.checked = all.length > 0 && n === all.length;
        selectAll.indeterminate = n > 0 && n < all.length;
    }
}

function confirmBulkDelete() {
    var count = document.querySelectorAll('.row-checkbox:checked').length;
    if (count === 0) { alert('Please select at least one location to delete.'); return false; }
    return confirm('Are you sure you want to delete ' + count + ' selected location(s)? Related prayer times may be affected. This cannot be undone.');
}

document.addEventListener('DOMContentLoaded', function() {
    if (document.getElementById('bulkDeleteForm')) updateBulkDeleteButton();
});
</script>
</body>
</html>
