<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';
$upload_results = [];

// সব লোকেশন লোড করুন (বাল্ক আপলোডের জন্য validation এর সময় ব্যবহার করবো)
$locations_stmt = $pdo->query("SELECT id, name, district FROM locations WHERE is_active = 1 ORDER BY district, name");
$locations = $locations_stmt->fetchAll(PDO::FETCH_ASSOC);

// Create location ID map for validation
$location_map = [];
foreach ($locations as $loc) {
    $location_map[$loc['id']] = $loc;
}

// হ্যান্ডেল ফাইল আপলোড
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv_file'])) {
    $file = $_FILES['csv_file'];
    
    // ফাইল ভ্যালিডেশন
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $message = '<div style="background: #fdd; color: #600; padding: 15px; border-radius: 6px; margin-bottom: 20px;">❌ ফাইল আপলোড ত্রুটি: ' . htmlspecialchars($file['error']) . '</div>';
    } elseif (pathinfo($file['name'], PATHINFO_EXTENSION) !== 'csv') {
        $message = '<div style="background: #fdd; color: #600; padding: 15px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: শুধুমাত্র CSV ফাইল আপলোড করুন</div>';
    } elseif ($file['size'] > 5 * 1024 * 1024) { // 5MB max
        $message = '<div style="background: #fdd; color: #600; padding: 15px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: ফাইলের সাইজ 5MB এর বেশি</div>';
    } else {
        // CSV পড়ুন এবং প্রসেস করুন
        $file_handle = fopen($file['tmp_name'], 'r');
        
        // হেডার লাইন পড়ুন এবং ভ্যালিডেট করুন
        $header = fgetcsv($file_handle);
        
        // প্রয়োজনীয় কলাম চেক করুন
        $required_columns = ['title', 'message', 'targeting_type'];
        $header_lower = array_map('strtolower', $header);
        
        $missing_columns = array_diff($required_columns, $header_lower);
        if (!empty($missing_columns)) {
            $message = '<div style="background: #fdd; color: #600; padding: 15px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: CSV ফাইলে নিম্নলিখিত প্রয়োজনীয় কলাম নেই: ' . htmlspecialchars(implode(', ', $missing_columns)) . '</div>';
            fclose($file_handle);
        } else {
            // ডেটা রো প্রসেস করুন
            $row_num = 2;
            $success_count = 0;
            $error_count = 0;
            $upload_results = [];
            
            // column index ম্যাপ তৈরি করুন
            $column_map = array_flip($header_lower);
            
            // প্রতিটি রো প্রসেস করুন
            while (($row_data = fgetcsv($file_handle)) !== false) {
                if (empty(array_filter($row_data))) { 
                    // খালি রো স্কিপ করুন
                    continue;
                }
                
                try {
                    // রো ডেটা এক্সট্র্যাক্ট করুন
                    $title = isset($column_map['title']) ? trim($row_data[$column_map['title']]) : '';
                    $message_text = isset($column_map['message']) ? trim($row_data[$column_map['message']]) : '';
                    $targeting_type = isset($column_map['targeting_type']) ? trim($row_data[$column_map['targeting_type']]) : '';
                    $location_ids_str = isset($column_map['location_ids']) ? trim($row_data[$column_map['location_ids']]) : '';
                    $image_url = isset($column_map['image_url']) ? trim($row_data[$column_map['image_url']]) : '';
                    $audio_url = isset($column_map['audio_url']) ? trim($row_data[$column_map['audio_url']]) : '';
                    $action_url = isset($column_map['action_url']) ? trim($row_data[$column_map['action_url']]) : '';
                    $is_scheduled = (isset($column_map['is_scheduled']) && strtolower(trim($row_data[$column_map['is_scheduled']])) === 'yes') ? 1 : 0;
                    $scheduled_time = isset($column_map['scheduled_time']) ? trim($row_data[$column_map['scheduled_time']]) : '';
                    $send_daily = (isset($column_map['send_daily']) && strtolower(trim($row_data[$column_map['send_daily']])) === 'yes') ? 1 : 0;
                    $is_active = (isset($column_map['is_active']) && strtolower(trim($row_data[$column_map['is_active']])) === 'yes') ? 1 : 0;
                    
                    // ভ্যালিডেশন
                    $errors = [];
                    
                    // টাইটেল চেক করুন
                    if (empty($title)) {
                        $errors[] = 'টাইটেল খালি';
                    } elseif (strlen($title) > 200) {
                        $errors[] = 'টাইটেল 200 অক্ষরের বেশি';
                    }
                    
                    // মেসেজ চেক করুন
                    if (empty($message_text)) {
                        $errors[] = 'মেসেজ খালি';
                    }
                    
                    // টার্গেটিং টাইপ চেক করুন
                    if (!in_array($targeting_type, ['all_locations', 'selected_locations'], true)) {
                        $errors[] = 'টার্গেটিং টাইপ অবৈধ (all_locations বা selected_locations হতে হবে)';
                    }
                    
                    // লোকেশন আইডি প্রসেস করুন এবং ভ্যালিডেট করুন
                    $location_ids = null;
                    if ($targeting_type === 'selected_locations') {
                        if (empty($location_ids_str)) {
                            $errors[] = 'selected_locations নির্বাচিত হয়েছে কিন্তু location_ids খালি';
                        } else {
                            // location_ids পার্স করুন (comma-separated)
                            $id_array = array_filter(array_map('intval', array_map('trim', explode(',', $location_ids_str))));
                            
                            if (empty($id_array)) {
                                $errors[] = 'location_ids পার্স করা যায়নি';
                            } else {
                                // সব location_ids ভ্যালিডেট করুন
                                $valid_ids = [];
                                $invalid_ids = [];
                                
                                foreach ($id_array as $loc_id) {
                                    if (isset($location_map[$loc_id])) {
                                        $valid_ids[] = $loc_id;
                                    } else {
                                        $invalid_ids[] = $loc_id;
                                    }
                                }
                                
                                if (!empty($invalid_ids)) {
                                    $errors[] = 'অবৈধ location_ids: ' . implode(', ', $invalid_ids);
                                }
                                
                                if (!empty($valid_ids)) {
                                    $location_ids = json_encode($valid_ids);
                                }
                            }
                        }
                    } else {
                        // all_locations এর জন্য location_ids ইগনোর করুন
                        $location_ids = null;
                    }
                    
                    // শিডিউল্ড সেটিংস ভ্যালিডেট করুন
                    if ($is_scheduled && empty($scheduled_time)) {
                        $errors[] = 'is_scheduled হয় কিন্তু scheduled_time খালি (HH:MM ফরম্যাটে থাকতে হবে)';
                    }
                    
                    if ($is_scheduled && !empty($scheduled_time)) {
                        // সময় ফরম্যাট চেক করুন (HH:MM)
                        if (!preg_match('/^\d{2}:\d{2}$/', $scheduled_time)) {
                            $errors[] = 'scheduled_time ফরম্যাট অবৈধ (HH:MM ফরম্যাটে থাকতে হবে যেমন ০৪:৩০)';
                        }
                    }
                    
                    // URL ফরম্যাট চেক করুন (যদি থাকে)
                    if (!empty($image_url) && !filter_var($image_url, FILTER_VALIDATE_URL)) {
                        $errors[] = 'image_url অবৈধ URL ফরম্যাট';
                    }
                    
                    if (!empty($audio_url) && !filter_var($audio_url, FILTER_VALIDATE_URL)) {
                        $errors[] = 'audio_url অবৈধ URL ফরম্যাট';
                    }
                    
                    if (!empty($action_url) && !filter_var($action_url, FILTER_VALIDATE_URL)) {
                        $errors[] = 'action_url অবৈধ URL ফরম্যাট';
                    }
                    
                    // যদি কোনো ত্রুটি থাকে তাহলে স্টোর করুন
                    if (!empty($errors)) {
                        $upload_results[] = [
                            'row' => $row_num,
                            'title' => htmlspecialchars($title),
                            'status' => 'error',
                            'errors' => $errors
                        ];
                        $error_count++;
                    } else {
                        // ডাটাবেসে ইনসার্ট করুন
                        $stmt = $pdo->prepare("
                            INSERT INTO notifications (
                                title, message, image_url, action_url, audio_url,
                                is_active, is_scheduled, scheduled_time, send_daily,
                                targeting_type, location_ids
                            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                        ");
                        
                        $stmt->execute([
                            $title,
                            $message_text,
                            !empty($image_url) ? $image_url : null,
                            !empty($action_url) ? $action_url : null,
                            !empty($audio_url) ? $audio_url : null,
                            $is_active,
                            $is_scheduled,
                            $is_scheduled ? $scheduled_time : null,
                            $send_daily,
                            $targeting_type,
                            $location_ids
                        ]);
                        
                        $upload_results[] = [
                            'row' => $row_num,
                            'title' => htmlspecialchars($title),
                            'status' => 'success'
                        ];
                        $success_count++;
                    }
                    
                } catch (Exception $e) {
                    $upload_results[] = [
                        'row' => $row_num,
                        'title' => htmlspecialchars($title ?? 'অজানা'),
                        'status' => 'error',
                        'errors' => [$e->getMessage()]
                    ];
                    $error_count++;
                }
                
                $row_num++;
            }
            
            fclose($file_handle);
            
            // সামারি মেসেজ দেখান
            $message = '<div style="background: #e8f5e9; border: 2px solid #4caf50; color: #1b5e20; padding: 15px; border-radius: 6px; margin-bottom: 20px;">';
            $message .= '<h3 style="margin-top: 0; margin-bottom: 10px;">✅ আপলোড সম্পন্ন!</h3>';
            $message .= '<p><strong>সফল:</strong> ' . $success_count . ' টি নোটিফিকেশন</p>';
            if ($error_count > 0) {
                $message .= '<p><strong>ব্যর্থ:</strong> ' . $error_count . ' টি নোটিফিকেশন</p>';
            }
            $message .= '</div>';
        }
    }
}

// CSV টেমপ্লেট ডাউনলোড করুন
if (isset($_GET['download_template'])) {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="notification_template.csv"');
    
    $output = fopen('php://output', 'w');
    
    // UTF-8 BOM যোগ করুন (Excel এর জন্য)
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // হেডার লিখুন
    fputcsv($output, [
        'title',
        'message',
        'targeting_type',
        'location_ids',
        'image_url',
        'audio_url',
        'action_url',
        'is_scheduled',
        'scheduled_time',
        'send_daily',
        'is_active'
    ]);
    
    // উদাহরণ রো লিখুন
    fputcsv($output, [
        'সেহরির সময় হয়েছে',
        'আস দিন সেহরির শেষ সময় হয়েছে। দ্রুত শেষ করুন এবং ওজু করুন।',
        'selected_locations',
        '1,2,3',
        'https://example.com/sehri.jpg',
        'https://example.com/notification.mp3',
        'https://example.com',
        'yes',
        '04:30',
        'yes',
        'yes'
    ]);
    
    // সব লোকেশন টার্গেটিং এর উদাহরণ
    fputcsv($output, [
        'ইফতারের সময় হয়েছে',
        'আস দিন ইফতারের সময় এসেছে। আল্লাহ আপনার রোজা কবুল করুন।',
        'all_locations',
        '',
        'https://example.com/iftar.jpg',
        'https://example.com/notification.mp3',
        '',
        'no',
        '',
        'no',
        'yes'
    ]);
    
    fclose($output);
    exit;
}

?>
<?php include 'header.php'; ?>

<div class="container">
    <h2>📦 নোটিফিকেশন বাল্ক আপলোড</h2>
    
    <?= $message ?>
    
    <div class="section" style="max-width: 800px;">
        <h3>📋 ধাপ ১: CSV টেমপ্লেট ডাউনলোড করুন</h3>
        <p style="margin-bottom: 15px;">প্রথমে CSV টেমপ্লেট ডাউনলোড করুন এবং আপনার নোটিফিকেশন ডেটা দিয়ে পূরণ করুন।</p>
        <a href="?download_template=1" class="btn" style="background: #4CAF50; color: white; display: inline-block; padding: 12px 25px; text-decoration: none; border-radius: 6px; font-weight: 600;">
            ⬇️ টেমপ্লেট ডাউনলোড করুন
        </a>
    </div>
    
    <div class="section" style="max-width: 800px;">
        <h3>📖 ধাপ ২: CSV ফাইল আপলোড করুন</h3>
        
        <form method="POST" enctype="multipart/form-data" style="margin-bottom: 30px;">
            <div class="form-group">
                <label>CSV ফাইল নির্বাচন করুন *</label>
                <input type="file" name="csv_file" accept=".csv" required style="padding: 12px; border: 2px dashed #007bff; border-radius: 6px; width: 100%; cursor: pointer;">
                <small style="color: #666; display: block; margin-top: 10px;">✓ শুধুমাত্র CSV ফাইল ( সর্বোচ্চ 5MB )</small>
            </div>
            <button type="submit" class="btn" style="background: #007bff; color: white; padding: 12px 25px; border: none; border-radius: 6px; cursor: pointer; font-weight: 600; font-size: 15px;">
                ⬆️ আপলোড করুন
            </button>
        </form>
    </div>
    
    <!-- আপলোড ফলাফল -->
    <?php if (!empty($upload_results)): ?>
        <div class="section">
            <h3>📊 আপলোড ফলাফল</h3>
            
            <div style="max-height: 500px; overflow-y: auto;">
                <table style="width: 100%; border-collapse: collapse;">
                    <thead>
                        <tr style="background: #f5f5f5; border-bottom: 2px solid #ddd;">
                            <th style="padding: 12px; text-align: left; font-weight: 600;">রো</th>
                            <th style="padding: 12px; text-align: left; font-weight: 600;">টাইটেল</th>
                            <th style="padding: 12px; text-align: left; font-weight: 600;">স্ট্যাটাস</th>
                            <th style="padding: 12px; text-align: left; font-weight: 600;">বার্তা</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($upload_results as $result): ?>
                            <tr style="border-bottom: 1px solid #eee;">
                                <td style="padding: 12px;"><?= $result['row'] ?></td>
                                <td style="padding: 12px;"><?= $result['title'] ?></td>
                                <td style="padding: 12px;">
                                    <?php if ($result['status'] === 'success'): ?>
                                        <span style="background: #d4edda; color: #155724; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 500;">✅ সফল</span>
                                    <?php else: ?>
                                        <span style="background: #f8d7da; color: #721c24; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 500;">❌ ব্যর্থ</span>
                                    <?php endif; ?>
                                </td>
                                <td style="padding: 12px;">
                                    <?php if (isset($result['errors']) && !empty($result['errors'])): ?>
                                        <ul style="margin: 0; padding-left: 20px; font-size: 13px; color: #721c24;">
                                            <?php foreach ($result['errors'] as $error): ?>
                                                <li><?= htmlspecialchars($error) ?></li>
                                            <?php endforeach; ?>
                                        </ul>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    <?php endif; ?>
    
    <!-- সাহায্য সেকশন -->
    <div class="section" style="background: #f0f8ff; border-left: 4px solid #2196F3; max-width: 800px;">
        <h3>❓ সাহায্য এবং বর্ণনা</h3>
        
        <h4 style="margin-top: 20px; margin-bottom: 10px;">📌 কলাম বর্ণনা:</h4>
        <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
            <tr style="background: #e3f2fd;">
                <th style="padding: 10px; text-align: left; border: 1px solid #90caf9;">কলামের নাম</th>
                <th style="padding: 10px; text-align: left; border: 1px solid #90caf9;">বর্ণনা</th>
                <th style="padding: 10px; text-align: left; border: 1px solid #90caf9;">উদাহরণ</th>
            </tr>
            <tr>
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>title *</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">নোটিফিকেশনের শিরোনাম (প্রয়োজনীয়)</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">সেহরির সময় হয়েছে</td>
            </tr>
            <tr style="background: #fafafa;">
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>message *</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">নোটিফিকেশনের বার্তা (প্রয়োজনীয়)</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">দ্রুত সেহরি শেষ করুন</td>
            </tr>
            <tr>
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>targeting_type *</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">নোটিফিকেশন পাঠানোর গন্তব্য</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">all_locations বা selected_locations</td>
            </tr>
            <tr style="background: #fafafa;">
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>location_ids</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">নির্দিষ্ট জেলার ID (যদি selected_locations হয় তাহলে প্রয়োজনীয়)</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">1,2,3 (কমা দিয়ে আলাদা করুন)</td>
            </tr>
            <tr>
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>image_url</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">নোটিফিকেশনের ছবি</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">https://example.com/image.jpg</td>
            </tr>
            <tr style="background: #fafafa;">
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>audio_url</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">কাস্টম সাউন্ড</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">https://example.com/notify.mp3</td>
            </tr>
            <tr>
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>action_url</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">ক্লিক করলে যাবে কোথায়</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">https://example.com</td>
            </tr>
            <tr style="background: #fafafa;">
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>is_scheduled</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">নির্দিষ্ট সময়ে পাঠাবেন কিনা</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">yes / no</td>
            </tr>
            <tr>
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>scheduled_time</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">সময় (HH:MM ফরম্যাটে, যদি is_scheduled = yes হয় তাহলে প্রয়োজনীয়)</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">04:30</td>
            </tr>
            <tr style="background: #fafafa;">
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>send_daily</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">প্রতিদিন পাঠাবেন কিনা</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">yes / no</td>
            </tr>
            <tr>
                <td style="padding: 10px; border: 1px solid #90caf9;"><strong>is_active</strong></td>
                <td style="padding: 10px; border: 1px solid #90caf9;">নোটিফিকেশন সক্রিয় রাখবেন কিনা</td>
                <td style="padding: 10px; border: 1px solid #90caf9;">yes / no</td>
            </tr>
        </table>
        
        <h4 style="margin-top: 20px; margin-bottom: 10px;">✅ গুরুত্বপূর্ণ নোট:</h4>
        <ul style="margin: 0; padding-left: 30px; line-height: 1.8;">
            <li><strong>প্রয়োজনীয় কলাম:</strong> title, message, targeting_type</li>
            <li><strong>targeting_type:</strong> 
                <ul style="margin: 10px 0; padding-left: 20px;">
                    <li><strong>all_locations:</strong> সব জেলায় নোটিফিকেশন পাঠান</li>
                    <li><strong>selected_locations:</strong> শুধু নির্দিষ্ট জেলায় পাঠান (location_ids প্রয়োজনীয়)</li>
                </ul>
            </li>
            <li><strong>location_ids:</strong> শুধুমাত্র selected_locations এর জন্য প্রয়োজনীয়, কমা দিয়ে আলাদা করুন</li>
            <li><strong>সময় ফরম্যাট:</strong> 24 ঘণ্টার ফরম্যাটে HH:MM (যেমন 04:30, 18:45)</li>
            <li><strong>হ্যাঁ/না ফিল্ড:</strong> শুধুমাত্র "yes" বা "no" লিখুন (কেস-ইনসেনসিটিভ)</li>
            <li><strong>URL:</strong> সম্পূর্ণ URL হতে হবে (https:// দিয়ে শুরু হতে হবে)</li>
            <li><strong>আপলোড সীমা:</strong> সর্বোচ্চ 5MB ফাইল সাইজ</li>
        </ul>
    </div>
    
    <div style="margin-top: 30px; text-align: center;">
        <a href="notifications.php" class="btn" style="background: #6c757d; color: white; display: inline-block; padding: 12px 25px; text-decoration: none; border-radius: 6px; font-weight: 600;">
            ← নোটিফিকেশন প্যানেলে ফিরুন
        </a>
    </div>
</div>

<style>
    .btn {
        transition: opacity 0.3s;
    }
    .btn:hover {
        opacity: 0.9;
    }
</style>
</body>
</html>
