<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';

// সব লোকেশন লোড করুন
$locations = $pdo->query("SELECT id, name, district FROM locations WHERE is_active = 1 ORDER BY district, name")->fetchAll();

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'add') {
        // সময়-নির্ধারিত নোটিফিকেশন কিনা চেক করুন
        $is_scheduled = isset($_POST['is_scheduled']) ? 1 : 0;
        $scheduled_time = ($is_scheduled && isset($_POST['scheduled_time'])) ? $_POST['scheduled_time'] : null;
        $send_daily = ($is_scheduled && isset($_POST['send_daily'])) ? 1 : 0;
        
        // লোকেশন টার্গেটিং প্রসেস করুন - ✅ STRICT VALIDATION
        $targeting_type = isset($_POST['targeting_type']) ? trim($_POST['targeting_type']) : '';
        $location_ids = null;
        $selected_ids = [];
        $is_valid = false;
        
        // ✅ MUST select either all_locations or selected_locations
        if (empty($targeting_type)) {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: অবশ্যই "সব লোকেশনে" অথবা "নির্দিষ্ট লোকেশনে" একটি বিকল্প নির্বাচন করতে হবে!</div>';
        } elseif (!in_array($targeting_type, ['all_locations', 'selected_locations'], true)) {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: অবৈধ টার্গেটিং টাইপ</div>';
        } elseif ($targeting_type === 'all_locations') {
            $is_valid = true;
        } elseif ($targeting_type === 'selected_locations') {
            // যদি selected_locations হয়, তাহলে অবশ্যই কিছু location নির্বাচন থাকতে হবে
            if (!isset($_POST['selected_locations']) || !is_array($_POST['selected_locations']) || empty($_POST['selected_locations'])) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: "নির্দিষ্ট লোকেশনে পাঠান" নির্বাচন করলে কমপক্ষে একটি জেলা/শহর চেক করতে হবে!</div>';
            } else {
                // নির্বাচিত location IDs validate করুন
                $selected_ids = array_filter(array_map('intval', $_POST['selected_locations']));
                
                if (empty($selected_ids)) {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: নির্বাচিত লোকেশন IDs বৈধ নয়</div>';
                } else {
                    // Verify সব নির্বাচিত IDs ডাটাবেসে আছে কিনা
                    $placeholders = implode(',', array_fill(0, count($selected_ids), '?'));
                    $check_stmt = $pdo->prepare("SELECT COUNT(*) as cnt FROM locations WHERE id IN ($placeholders) AND is_active = 1");
                    $check_stmt->execute($selected_ids);
                    $result = $check_stmt->fetch();
                    
                    if ($result['cnt'] == count($selected_ids)) {
                        $location_ids = json_encode($selected_ids);
                        $is_valid = true;
                    } else {
                        $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: কিছু নির্বাচিত লোকেশন ডাটাবেসে নেই বা inactive</div>';
                    }
                }
            }
        }
        
        // ✅ শুধুমাত্র validation pass হলেই insert করুন
        if ($is_valid) {
            $stmt = $pdo->prepare("
                INSERT INTO notifications (title, message, image_url, action_url, audio_url, is_active, is_scheduled, scheduled_time, send_daily, targeting_type, location_ids)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            try {
                $stmt->execute([
                    $_POST['title'],
                    $_POST['message'],
                    $_POST['image_url'] ?: null,
                    $_POST['action_url'] ?: null,
                    $_POST['audio_url'] ?: null,
                    isset($_POST['is_active']) ? 1 : 0,
                    $is_scheduled,
                    $scheduled_time,
                    $send_daily,
                    $targeting_type,
                    $location_ids,
                ]);
                
                // Success message with targeting info
                $targeting_info = $targeting_type === 'all_locations' 
                    ? '🌍 সব জেলায়' 
                    : '📌 ' . count($selected_ids) . 'টি জেলায় (IDs: ' . implode(', ', $selected_ids) . ')';
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✅ নোটিফিকেশন সফলভাবে যোগ হয়েছে! [' . $targeting_info . ']</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">❌ ত্রুটি: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
        }
    }
    
    // Delete করার অ্যাকশন
    if (isset($_POST['action']) && $_POST['action'] === 'delete') {
        $stmt = $pdo->prepare("DELETE FROM notifications WHERE id = ?");
        
        try {
            $stmt->execute([$_POST['id']]);
            $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✓ নোটিফিকেশন সফলভাবে মুছে দেওয়া হয়েছে!</div>';
        } catch (Exception $e) {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✗ ত্রুটি: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    // বাল্ক Delete করার অ্যাকশন
    if (isset($_POST['action']) && $_POST['action'] === 'bulk_delete') {
        if (isset($_POST['selected_ids']) && is_array($_POST['selected_ids']) && !empty($_POST['selected_ids'])) {
            $ids = array_filter(array_map('intval', $_POST['selected_ids']));
            
            if (!empty($ids)) {
                try {
                    $placeholders = implode(',', array_fill(0, count($ids), '?'));
                    $stmt = $pdo->prepare("DELETE FROM notifications WHERE id IN ($placeholders)");
                    $stmt->execute($ids);
                    $deleted_count = count($ids);
                    $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✓ ' . $deleted_count . ' টি নোটিফিকেশন সফলভাবে মুছে দেওয়া হয়েছে!</div>';
                } catch (Exception $e) {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✗ ত্রুটি: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
            } else {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✗ কোনো বৈধ নোটিফিকেশন নির্বাচন করা হয়নি</div>';
            }
        } else {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✗ কমপক্ষে একটি নোটিফিকেশন নির্বাচন করুন</div>';
        }
    }
    
    // স্ট্যাটাস টগল করার অ্যাকশন
    if (isset($_POST['action']) && $_POST['action'] === 'toggle_status') {
        $stmt = $pdo->prepare("UPDATE notifications SET is_active = NOT is_active WHERE id = ?");
        
        try {
            $stmt->execute([$_POST['id']]);
            $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✓ নোটিফিকেশন স্ট্যাটাস আপডেট হয়েছে!</div>';
        } catch (Exception $e) {
            $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">✗ ত্রুটি: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// Get notifications with location info
$notifications_query = "SELECT * FROM notifications ORDER BY created_at DESC";
$notifications_result = $pdo->query($notifications_query)->fetchAll();

// Enhance notifications with location names
$notifications = [];
foreach ($notifications_result as $notif) {
    $notif['location_names'] = [];
    
    if ($notif['targeting_type'] === 'selected_locations' && !empty($notif['location_ids'])) {
        try {
            $selected_ids = json_decode($notif['location_ids'], true);
            if (is_array($selected_ids) && !empty($selected_ids)) {
                $placeholders = implode(',', array_fill(0, count($selected_ids), '?'));
                $loc_stmt = $pdo->prepare("SELECT name FROM locations WHERE id IN ($placeholders)");
                $loc_stmt->execute($selected_ids);
                $notif['location_names'] = array_column($loc_stmt->fetchAll(), 'name');
            }
        } catch (Exception $e) {
            // Silent error handling
        }
    }
    
    $notifications[] = $notif;
}
?>
<?php include 'header.php'; ?>

<div class="container">
    <h2>📬 নোটিফিকেশন ব্যবস্থাপনা</h2>
    
    <?= $message ?>
    
    <!-- বাল্ক আপলোড অপশন -->
    <div style="background: #fff3e0; border: 2px solid #ff9800; border-radius: 8px; padding: 15px; margin-bottom: 25px; display: flex; justify-content: space-between; align-items: center;">
        <div>
            <h3 style="margin: 0 0 5px 0; color: #e65100;">📦 দ্রুত নোটিফিকেশন যোগ করুন</h3>
            <p style="margin: 5px 0; color: #bf360c; font-size: 14px;">অনেক নোটিফিকেশন একসাথে যোগ করতে CSV ফাইল আপলোড করুন</p>
        </div>
        <a href="notification_bulk_upload.php" class="btn" style="background: #ff9800; color: white; padding: 12px 30px; text-decoration: none; border-radius: 6px; font-weight: 600; white-space: nowrap; margin-left: 20px;">
            ⬆️ বাল্ক আপলোড
        </a>
    </div>
    
    <div class="section">
        <h3>নতুন নোটিফিকেশন যোগ করুন</h3>
        <form method="POST" style="max-width: 600px;" onsubmit="return validateLocationForm();">
            <input type="hidden" name="action" value="add">
            <div class="form-group">
                <label>শিরোনাম *</label>
                <input type="text" name="title" placeholder="যেমন: সেহরির সময় হয়েছে" required>
            </div>
            <div class="form-group">
                <label>বার্তা *</label>
                <textarea name="message" rows="4" placeholder="নোটিফিকেশনের বিবরণ লিখুন" required></textarea>
            </div>
            <div class="form-group">
                <label>ছবির URL</label>
                <input type="url" name="image_url" placeholder="https://example.com/image.jpg">
            </div>
            <div class="form-group">
                <label>অডিও URL (কাস্টম সাউন্ড)</label>
                <input type="url" name="audio_url" placeholder="https://example.com/notification.mp3">
                <small style="color: #666; display: block; margin-top: 5px;">MP3 বা WAV ফাইলের URL। এই সাউন্ড নোটিফিকেশনের সাথে বাজবে।</small>
            </div>
            <div class="form-group">
                <label>অ্যাকশন URL</label>
                <input type="url" name="action_url" placeholder="https://example.com">
            </div>
            
            <!-- 📍 লোকেশন টার্গেটিং সেকশন -->
            <div class="form-group" style="background: #f0f8ff; padding: 15px; border-radius: 6px; border-left: 4px solid #FF6B6B;">
                <label style="font-size: 16px; font-weight: 600; margin-bottom: 15px; display: block;">📍 নোটিফিকেশন পাঠাবেন কোথায়? *</label>
                
                <div style="display: flex; gap: 20px; margin-bottom: 15px;">
                    <label style="display: flex; align-items: center; cursor: pointer;">
                        <input type="radio" name="targeting_type" value="all_locations" id="all_locations" style="width: 20px; height: 20px; margin-right: 8px; cursor: pointer;">
                        <span style="font-size: 15px;">🌍 সব লোকেশনে পাঠান (সমস্ত জেলা/শহর)</span>
                    </label>
                </div>
                
                <div style="display: flex; gap: 20px; margin-bottom: 15px;">
                    <label style="display: flex; align-items: center; cursor: pointer;">
                        <input type="radio" name="targeting_type" value="selected_locations" id="selected_locations" style="width: 20px; height: 20px; margin-right: 8px; cursor: pointer;">
                        <span style="font-size: 15px;">📌 নির্দিষ্ট লোকেশনে পাঠান (শুধু নির্বাচিত) *</span>
                    </label>
                </div>
                
                <div id="location_selector" style="display: none; margin-top: 15px;">
                    <label style="display: block; margin-bottom: 10px; font-weight: 600;">যে লোকেশন গুলোতে পাঠাবেন তা নির্বাচন করুন:</label>
                    <div style="background: white; border: 2px solid #ddd; border-radius: 6px; padding: 15px; max-height: 300px; overflow-y: auto;">
                        <?php 
                        // জেলা অনুযায়ী লোকেশন সংগঠিত করুন
                        $locations_by_district = [];
                        foreach ($locations as $loc) {
                            $district = $loc['district'];
                            if (!isset($locations_by_district[$district])) {
                                $locations_by_district[$district] = [];
                            }
                            $locations_by_district[$district][] = $loc;
                        }
                        
                        foreach ($locations_by_district as $district => $district_locations): 
                        ?>
                            <div style="margin-bottom: 15px;">
                                <div style="font-weight: 600; color: #333; margin-bottom: 8px; padding-bottom: 5px; border-bottom: 2px solid #007bff;">
                                    📍 <?= htmlspecialchars($district) ?>
                                </div>
                                <div style="padding-left: 10px;">
                                    <?php foreach ($district_locations as $loc): ?>
                                        <label style="display: flex; align-items: center; margin-bottom: 8px; cursor: pointer;">
                                            <input type="checkbox" name="selected_locations[]" value="<?= $loc['id'] ?>" style="width: 18px; height: 18px; margin-right: 8px; cursor: pointer;">
                                            <span style="font-size: 14px;"><?= htmlspecialchars($loc['name']) ?></span>
                                        </label>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <small style="color: #666; display: block; margin-top: 10px;">
                        ✓ টিপস: কোন লোকেশন নির্বাচন করবেন না মানে সব লোকেশনে পাঠাবে।
                    </small>
                </div>
            </div>
            
            <!-- সময়-নির্ধারিত নোটিফিকেশন অপশন -->
            <div class="form-group" style="background: #f9f9f9; padding: 15px; border-radius: 6px; border-left: 4px solid #007bff;">
                <label style="display: flex; align-items: center; margin-bottom: 10px;">
                    <input type="checkbox" name="is_scheduled" id="is_scheduled" style="margin-right: 10px; width: 20px; height: 20px; cursor: pointer;">
                    <span style="cursor: pointer; font-size: 16px; font-weight: 600;">⏰ নির্দিষ্ট সময়ে পাঠান</span>
                </label>
                
                <div id="scheduled_options" style="display: none; margin-top: 10px;">
                    <div class="form-group">
                        <label>সময় নির্বাচন করুন *</label>
                        <input type="time" name="scheduled_time" id="scheduled_time" placeholder="HH:MM" style="padding: 10px; border: 1px solid #ddd; border-radius: 4px; width: 100%;">
                        <small style="color: #666; display: block; margin-top: 5px;">যেমন: 04:30 (সকাল ৪:৩০) বা 18:45 (সন্ধ্যা ৬:৪৫)</small>
                    </div>
                    
                    <div class="form-group">
                        <label style="display: flex; align-items: center;">
                            <input type="checkbox" name="send_daily" id="send_daily" style="margin-right: 10px; width: 20px; height: 20px; cursor: pointer;">
                            <span style="cursor: pointer;">প্রতিদিন এই সময়ে পাঠান</span>
                        </label>
                        <small style="color: #666; display: block; margin-top: 5px;">চেক করলে প্রতিদিন নির্দিষ্ট সময়ে নোটিফিকেশন পাঠানো হবে</small>
                    </div>
                </div>
            </div>
            
            <div class="form-group">
                <label>
                    <input type="checkbox" name="is_active" checked> সক্রিয় রাখুন
                </label>
            </div>
            <button type="submit" class="btn" style="background: #007bff; color: white;">✓ নোটিফিকেশন যোগ করুন</button>
        </form>
    </div>
    
    <div class="section">
        <h3>সব নোটিফিকেশন</h3>
        
        <!-- বাল্ক ডিলিট ফর্ম -->
        <form method="POST" id="bulk_delete_form" style="margin-bottom: 20px;">
            <input type="hidden" name="action" value="bulk_delete">
            <div style="display: flex; gap: 10px; align-items: center; margin-bottom: 15px;">
                <button type="button" id="bulk_delete_btn" class="btn" style="background: #dc3545; color: white; display: none;" onclick="bulkDelete()">
                    🗑️ নির্বাচিত মুছুন
                </button>
                <span id="selected_count" style="color: #666; font-weight: 500;">০টি নির্বাচিত</span>
                <button type="button" id="clear_selection_btn" class="btn" style="background: #6c757d; color: white; display: none;" onclick="clearSelection()">
                    ✕ নির্বাচন বাতিল করুন
                </button>
            </div>
            
            <table>
                <thead>
                    <tr>
                        <th style="width: 40px;"><input type="checkbox" id="select_all" title="সব নির্বাচন করুন"></th>
                        <th>শিরোনাম</th>
                        <th>বার্তা</th>
                        <th>তৈরি</th>
                        <th>টার্গেট</th>
                        <th>ধরন</th>
                        <th>স্ট্যাটাস</th>
                        <th>অ্যাকশন</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($notifications as $notif): ?>
                    <tr>
                        <td style="text-align: center;"><input type="checkbox" class="row_checkbox" data-id="<?= $notif['id'] ?>"></td>
                        <td><?= htmlspecialchars($notif['title']) ?></td>
                        <td><?= htmlspecialchars(substr($notif['message'], 0, 50)) ?>...</td>
                        <td><?= htmlspecialchars(date('d/m/Y H:i', strtotime($notif['created_at']))) ?></td>
                        <td>
                            <?php 
                            if ($notif['targeting_type'] === 'all_locations') {
                                echo '<span title="সব লোকেশনে পাঠানো হবে" style="background: #e3f2fd; color: #1976d2; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 500;">🌍 সব জেলা</span>';
                            } else {
                                if (!empty($notif['location_names'])) {
                                    $loc_text = implode(', ', array_slice($notif['location_names'], 0, 2));
                                    if (count($notif['location_names']) > 2) {
                                        $loc_text .= ' +' . (count($notif['location_names']) - 2);
                                    }
                                    echo '<span title="' . htmlspecialchars(implode(', ', $notif['location_names'])) . '" style="background: #fff3e0; color: #f57c00; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 500;">📍 ' . htmlspecialchars($loc_text) . '</span>';
                                } else {
                                    echo '<span style="background: #f3e5f5; color: #7b1fa2; padding: 4px 10px; border-radius: 12px; font-size: 12px; font-weight: 500;">📍 নির্দিষ্ট জেলা</span>';
                                }
                            }
                            ?>
                        </td>
                        <td>
                            <?php 
                            if ($notif['is_scheduled']) {
                                $time = $notif['scheduled_time'] ?? 'N/A';
                                $daily = $notif['send_daily'] ? '📅' : '⏱️';
                                echo '<span title="নির্দিষ্ট সময়ে পাঠানো">' . $daily . ' ' . $time . '</span>';
                            } else {
                                echo '<span title="এখনই পাঠ্য">🔔 তাৎক্ষণিক</span>';
                            }
                            ?>
                        </td>
                        <td>
                            <span class="status-badge <?= $notif['is_active'] ? 'active' : 'inactive' ?>">
                                <?= $notif['is_active'] ? '✓ সক্রিয়' : '✗ নিষ্ক্রিয়' ?>
                            </span>
                        </td>
                        <td class="actions">
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="action" value="toggle_status">
                                <input type="hidden" name="id" value="<?= $notif['id'] ?>">
                                <button type="submit" class="btn-small btn-toggle" title="<?= $notif['is_active'] ? 'নিষ্ক্রিয় করুন' : 'সক্রিয় করুন' ?>">
                                    <?= $notif['is_active'] ? '⏸️' : '▶️' ?>
                                </button>
                            </form>
                            <form method="POST" style="display: inline;" onsubmit="return confirm('আপনি কি এই নোটিফিকেশনটি মুছে দিতে নিশ্চিত?');">
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="id" value="<?= $notif['id'] ?>">
                                <button type="submit" class="btn-small btn-delete" title="মুছে দিন">🗑️</button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
            </table>
        </form>
    </div>
</div>

<style>
    .form-group { margin-bottom: 15px; }
    .form-group label { display: block; margin-bottom: 5px; font-weight: 500; }
    .form-group input, .form-group textarea { 
        width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; 
    }
    .form-group textarea { font-family: inherit; }
    
    /* Status badges */
    .status-badge {
        padding: 4px 10px;
        border-radius: 12px;
        font-size: 12px;
        font-weight: 500;
    }
    .status-badge.active {
        background: #d4edda;
        color: #155724;
    }
    .status-badge.inactive {
        background: #f8d7da;
        color: #721c24;
    }
    
    /* Action buttons */
    .actions {
        white-space: nowrap;
    }
    .btn-small {
        padding: 6px 10px;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        font-size: 14px;
        margin: 0 2px;
        transition: opacity 0.2s;
    }
    .btn-small:hover {
        opacity: 0.8;
    }
    .btn-toggle {
        background: #e7f3ff;
    }
    .btn-delete {
        background: #ffe7e7;
    }
    .btn-delete:hover {
        background: #ffcccc;
    }
    
    /* নতুন সিএসএস স্টাইল */
    .btn {
        padding: 10px 20px;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        font-size: 14px;
        font-weight: 600;
    }
    
    #scheduled_options {
        animation: slideDown 0.3s ease-in-out;
    }
    
    @keyframes slideDown {
        from {
            opacity: 0;
            transform: translateY(-10px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
</style>

<!-- JavaScript: সময় নির্বাচন অপশন দেখান/লুকান এবং লোকেশন সিলেক্টর -->
<script>
// ফর্ম validation function
function validateLocationForm() {
    const allLocationsRadio = document.getElementById('all_locations');
    const selectedLocationsRadio = document.getElementById('selected_locations');
    const locationCheckboxes = document.querySelectorAll('input[name="selected_locations[]"]');
    
    // Check if any radio button is selected
    if (!allLocationsRadio.checked && !selectedLocationsRadio.checked) {
        alert('❌ ত্রুটি: লোকেশন টার্গেটিং নির্বাচন করুন! (সব জেলা বা নির্দিষ্ট জেলা)');
        return false;
    }
    
    // If selected_locations is chosen, verify at least one location is checked
    if (selectedLocationsRadio.checked) {
        const checkedLocations = Array.from(locationCheckboxes).filter(cb => cb.checked);
        if (checkedLocations.length === 0) {
            alert('❌ ত্রুটি: "নির্দিষ্ট লোকেশনে পাঠান" নির্বাচন করেছেন কিন্তু কোনো লোকেশন চেক করেননি!\n\nঅন্তত একটি লোকেশন নির্বাচন করুন অথবা "সব লোকেশনে পাঠান" নির্বাচন করুন।');
            return false;
        }
    }
    
    return true; // Allow form submission
}

// Schedule checkbox toggle
document.getElementById('is_scheduled')?.addEventListener('change', function() {
    const scheduledOptions = document.getElementById('scheduled_options');
    const scheduledTimeInput = document.getElementById('scheduled_time');
    
    if (this.checked) {
        scheduledOptions.style.display = 'block';
        scheduledTimeInput.required = true;
    } else {
        scheduledOptions.style.display = 'none';
        scheduledTimeInput.required = false;
        scheduledTimeInput.value = '';
    }
});

// Location Targeting Radio Buttons
const all_locations_radio = document.getElementById('all_locations');
const selected_locations_radio = document.getElementById('selected_locations');
const location_selector = document.getElementById('location_selector');

// Function to toggle location selector visibility
function toggleLocationSelector() {
    if (selected_locations_radio && selected_locations_radio.checked) {
        location_selector.style.display = 'block';
    } else if (location_selector) {
        location_selector.style.display = 'none';
    }
}

// Add event listeners to radio buttons
if (all_locations_radio) {
    all_locations_radio.addEventListener('change', toggleLocationSelector);
}
if (selected_locations_radio) {
    selected_locations_radio.addEventListener('change', toggleLocationSelector);
}

// Initialize on page load
window.addEventListener('DOMContentLoaded', function() {
    toggleLocationSelector();
});

// Also run on page load (in case DOM is already ready)
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', toggleLocationSelector);
} else {
    toggleLocationSelector();
}

// ======= বাল্ক ডিলিট ফাংশনালিটি =======

// নির্বাচিত সংখ্যা আপডেট করার ফাংশন
function updateSelectedCount() {
    const checkboxes = document.querySelectorAll('input.row_checkbox:checked');
    const count = checkboxes.length;
    const countSpan = document.getElementById('selected_count');
    const bulkDeleteBtn = document.getElementById('bulk_delete_btn');
    const clearSelectionBtn = document.getElementById('clear_selection_btn');
    
    countSpan.textContent = count + 'টি নির্বাচিত';
    
    if (count > 0) {
        bulkDeleteBtn.style.display = 'inline-block';
        clearSelectionBtn.style.display = 'inline-block';
    } else {
        bulkDeleteBtn.style.display = 'none';
        clearSelectionBtn.style.display = 'none';
    }
}

// সব checkbox নির্বাচন করার ফাংশন
function toggleSelectAll() {
    const selectAllCheckbox = document.getElementById('select_all');
    const rowCheckboxes = document.querySelectorAll('input.row_checkbox');
    
    rowCheckboxes.forEach(checkbox => {
        checkbox.checked = selectAllCheckbox.checked;
    });
    
    updateSelectedCount();
}

// সব নির্বাচন বাতিল করার ফাংশন
function clearSelection() {
    const selectAllCheckbox = document.getElementById('select_all');
    const rowCheckboxes = document.querySelectorAll('input.row_checkbox');
    
    selectAllCheckbox.checked = false;
    rowCheckboxes.forEach(checkbox => {
        checkbox.checked = false;
    });
    
    updateSelectedCount();
}

// বাল্ক ডিলিট ফাংশন - সম্পূর্ণরূপে ফিক্সড
function bulkDelete() {
    // সব selected checkboxes খুঁজা
    const selectedCheckboxes = document.querySelectorAll('input.row_checkbox:checked');
    
    if (selectedCheckboxes.length === 0) {
        alert('❌ কমপক্ষে একটি নোটিফিকেশন নির্বাচন করুন');
        return false;
    }
    
    // Confirm message
    const confirmMsg = selectedCheckboxes.length === 1 
        ? 'এই ১টি নোটিফিকেশন মুছতে চান?' 
        : 'এই ' + selectedCheckboxes.length + 'টি নোটিফিকেশন মুছতে চান?';
    
    if (!confirm(confirmMsg)) {
        return false;
    }
    
    // নতুন form তৈরি করা
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = '';
    
    // Action field
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'action';
    actionInput.value = 'bulk_delete';
    form.appendChild(actionInput);
    
    // প্রতিটি selected checkbox এর ID যোগ করা
    selectedCheckboxes.forEach(checkbox => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'selected_ids[]';
        input.value = checkbox.getAttribute('data-id');
        form.appendChild(input);
    });
    
    // Form submit করা
    document.body.appendChild(form);
    form.submit();
    
    return false;
}

// সব checkbox এ event listener যোগ করা
document.addEventListener('DOMContentLoaded', function() {
    console.log('✓ Page fully loaded, initializing bulk delete feature');
    
    // Select All checkbox
    const selectAllCheckbox = document.getElementById('select_all');
    if (selectAllCheckbox) {
        selectAllCheckbox.addEventListener('change', function() {
            console.log('Select All checkbox changed:', this.checked);
            toggleSelectAll();
        });
    }
    
    // Individual row checkboxes
    const rowCheckboxes = document.querySelectorAll('input.row_checkbox');
    console.log('Found ' + rowCheckboxes.length + ' row checkboxes');
    
    rowCheckboxes.forEach(checkbox => {
        checkbox.addEventListener('change', function() {
            console.log('Row checkbox changed:', this.getAttribute('data-id'), 'Checked:', this.checked);
            updateSelectedCount();
        });
    });
    
    // প্রাথমিক count আপডেট করুন
    updateSelectedCount();
    console.log('Initial count updated');
});
</script>
</body>
</html>
