<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';

// Restore flash message after redirect (POST-Redirect-GET)
if (isset($_SESSION['prayer_times_message'])) {
    $message = $_SESSION['prayer_times_message'];
    unset($_SESSION['prayer_times_message']);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'add') {
            $stmt = $pdo->prepare("
                INSERT INTO prayer_times 
                (location_id, date, fajr, sunrise, dhuhr, asr, maghrib, isha, sehri_end, iftar, ramadan_day)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    fajr = VALUES(fajr),
                    sunrise = VALUES(sunrise),
                    dhuhr = VALUES(dhuhr),
                    asr = VALUES(asr),
                    maghrib = VALUES(maghrib),
                    isha = VALUES(isha),
                    sehri_end = VALUES(sehri_end),
                    iftar = VALUES(iftar),
                    ramadan_day = VALUES(ramadan_day)
            ");
            
            try {
                $stmt->execute([
                    $_POST['location_id'],
                    $_POST['date'],
                    $_POST['fajr'],
                    $_POST['sunrise'],
                    $_POST['dhuhr'],
                    $_POST['asr'],
                    $_POST['maghrib'],
                    $_POST['isha'],
                    $_POST['sehri_end'],
                    $_POST['iftar'],
                    $_POST['ramadan_day'],
                ]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Prayer time added successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['prayer_times_message'] = $message;
            header('Location: prayer-times.php');
            exit;
        } elseif ($_POST['action'] === 'update') {
            // Update existing entry
            $stmt = $pdo->prepare("
                UPDATE prayer_times 
                SET location_id = ?, date = ?, fajr = ?, sunrise = ?, dhuhr = ?, asr = ?, 
                    maghrib = ?, isha = ?, sehri_end = ?, iftar = ?, ramadan_day = ?
                WHERE id = ?
            ");
            
            try {
                $stmt->execute([
                    $_POST['location_id'],
                    $_POST['date'],
                    $_POST['fajr'],
                    $_POST['sunrise'],
                    $_POST['dhuhr'],
                    $_POST['asr'],
                    $_POST['maghrib'],
                    $_POST['isha'],
                    $_POST['sehri_end'],
                    $_POST['iftar'],
                    $_POST['ramadan_day'],
                    $_POST['id'],
                ]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Prayer time updated successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['prayer_times_message'] = $message;
            header('Location: prayer-times.php');
            exit;
        } elseif ($_POST['action'] === 'delete') {
            // Delete entry
            $stmt = $pdo->prepare("DELETE FROM prayer_times WHERE id = ?");
            try {
                $stmt->execute([$_POST['id']]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Prayer time deleted successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['prayer_times_message'] = $message;
            header('Location: prayer-times.php');
            exit;
        } elseif ($_POST['action'] === 'bulk_delete') {
            // Bulk delete entries
            if (isset($_POST['selected_ids']) && is_array($_POST['selected_ids']) && !empty($_POST['selected_ids'])) {
                $ids = array_map('intval', $_POST['selected_ids']);
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $stmt = $pdo->prepare("DELETE FROM prayer_times WHERE id IN ($placeholders)");
                
                try {
                    $stmt->execute($ids);
                    $deletedCount = $stmt->rowCount();
                    $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Successfully deleted ' . $deletedCount . ' prayer time entry/entries!</div>';
                } catch (Exception $e) {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
            } else {
                $message = '<div style="background: #ffd; color: #660; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please select at least one entry to delete.</div>';
            }
            $_SESSION['prayer_times_message'] = $message;
            header('Location: prayer-times.php');
            exit;
        }
    }
}

// Get locations
$locations = $pdo->query("SELECT id, name, district FROM locations WHERE is_active = 1 ORDER BY district, name")->fetchAll();

// Get prayer times
$prayerTimes = $pdo->query("
    SELECT pt.*, l.name as location_name, l.district
    FROM prayer_times pt
    JOIN locations l ON pt.location_id = l.id
    ORDER BY pt.date DESC, l.name
    LIMIT 100
")->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Prayer Times - Admin</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container">
        <h2>Manage Prayer Times</h2>
        
        <?= $message ?>
        
        <div class="section">
            <h3>Add New Prayer Time</h3>
            <form method="POST" style="max-width: 600px;">
                <input type="hidden" name="action" value="add">
                <div class="form-group">
                    <label>Location *</label>
                    <select name="location_id" required style="width: 100%; padding: 8px;">
                        <?php foreach ($locations as $loc): ?>
                            <option value="<?= $loc['id'] ?>"><?= htmlspecialchars($loc['name'] . ', ' . $loc['district']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>Date * (YYYY-MM-DD)</label>
                    <input type="date" name="date" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Fajr * (HH:MM)</label>
                        <input type="time" name="fajr" required>
                    </div>
                    <div class="form-group">
                        <label>Sunrise * (HH:MM)</label>
                        <input type="time" name="sunrise" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Dhuhr * (HH:MM)</label>
                        <input type="time" name="dhuhr" required>
                    </div>
                    <div class="form-group">
                        <label>Asr * (HH:MM)</label>
                        <input type="time" name="asr" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Maghrib * (HH:MM)</label>
                        <input type="time" name="maghrib" required>
                    </div>
                    <div class="form-group">
                        <label>Isha * (HH:MM)</label>
                        <input type="time" name="isha" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Sehri End * (HH:MM)</label>
                        <input type="time" name="sehri_end" required>
                    </div>
                    <div class="form-group">
                        <label>Iftar * (HH:MM)</label>
                        <input type="time" name="iftar" required>
                    </div>
                </div>
                <div class="form-group">
                    <label>Ramadan Day *</label>
                    <input type="number" name="ramadan_day" min="1" max="30" required>
                </div>
                <button type="submit" class="btn">Add Prayer Time</button>
            </form>
        </div>
        
        <div class="section">
            <h3>Recent Prayer Times</h3>
            <?php if (empty($prayerTimes)): ?>
                <p style="color: #666; padding: 20px; text-align: center;">No prayer times found. Add entries using the form above.</p>
            <?php else: ?>
                <!-- Hidden form for single delete (used via JS to avoid nested forms) -->
                <form method="POST" id="singleDeleteForm" style="display: none;">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" id="singleDeleteId">
                </form>

                <!-- Bulk Delete Form -->
                <form method="POST" id="bulkDeleteForm" onsubmit="return confirmBulkDelete();">
                    <input type="hidden" name="action" value="bulk_delete">
                    <div style="margin-bottom: 15px;">
                        <button type="submit" id="bulkDeleteBtn" class="btn btn-danger" disabled style="background: #dc3545; color: white; padding: 10px 20px; border: none; border-radius: 6px; cursor: not-allowed; font-weight: 600; opacity: 0.6;">
                            Delete Selected (<span id="selectedCount">0</span>)
                        </button>
                    </div>
                    
                    <div style="overflow-x: auto;">
                        <table>
                            <thead>
                                <tr>
                                    <th style="width: 50px;">
                                        <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                                    </th>
                                    <th>Date</th>
                                    <th>Location</th>
                                    <th>Fajr</th>
                                    <th>Sunrise</th>
                                    <th>Dhuhr</th>
                                    <th>Asr</th>
                                    <th>Maghrib</th>
                                    <th>Isha</th>
                                    <th>Sehri</th>
                                    <th>Iftar</th>
                                    <th>Ramadan Day</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($prayerTimes as $pt): ?>
                                    <tr>
                                        <td style="text-align: center;">
                                            <input type="checkbox" name="selected_ids[]" value="<?= $pt['id'] ?>" class="row-checkbox" onchange="updateBulkDeleteButton()">
                                        </td>
                                        <td><?= htmlspecialchars($pt['date']) ?></td>
                                        <td><?= htmlspecialchars($pt['location_name']) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['fajr'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['sunrise'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['dhuhr'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['asr'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['maghrib'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['isha'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['sehri_end'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($pt['iftar'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars($pt['ramadan_day']) ?></td>
                                        <td>
                                            <button type="button" onclick="editEntry(<?= htmlspecialchars(json_encode($pt)) ?>)" class="btn-small" style="margin-right: 5px;">Edit</button>
                                            <button type="button" class="btn-small btn-danger" onclick="deleteEntry(<?= (int)$pt['id'] ?>)">Delete</button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Edit Modal -->
    <div id="editModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000; align-items: center; justify-content: center;">
        <div style="background: white; padding: 30px; border-radius: 12px; max-width: 600px; width: 90%; max-height: 90vh; overflow-y: auto;">
            <h3 style="margin-bottom: 20px;">Edit Prayer Time</h3>
            <form method="POST" id="editForm">
                <input type="hidden" name="action" value="update">
                <input type="hidden" name="id" id="edit_id">
                <div class="form-group">
                    <label>Location *</label>
                    <select name="location_id" id="edit_location_id" required style="width: 100%; padding: 8px;">
                        <?php foreach ($locations as $loc): ?>
                            <option value="<?= $loc['id'] ?>"><?= htmlspecialchars($loc['name'] . ', ' . $loc['district']) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="form-group">
                    <label>Date * (YYYY-MM-DD)</label>
                    <input type="date" name="date" id="edit_date" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Fajr * (HH:MM)</label>
                        <input type="time" name="fajr" id="edit_fajr" required>
                    </div>
                    <div class="form-group">
                        <label>Sunrise * (HH:MM)</label>
                        <input type="time" name="sunrise" id="edit_sunrise" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Dhuhr * (HH:MM)</label>
                        <input type="time" name="dhuhr" id="edit_dhuhr" required>
                    </div>
                    <div class="form-group">
                        <label>Asr * (HH:MM)</label>
                        <input type="time" name="asr" id="edit_asr" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Maghrib * (HH:MM)</label>
                        <input type="time" name="maghrib" id="edit_maghrib" required>
                    </div>
                    <div class="form-group">
                        <label>Isha * (HH:MM)</label>
                        <input type="time" name="isha" id="edit_isha" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Sehri End * (HH:MM)</label>
                        <input type="time" name="sehri_end" id="edit_sehri_end" required>
                    </div>
                    <div class="form-group">
                        <label>Iftar * (HH:MM)</label>
                        <input type="time" name="iftar" id="edit_iftar" required>
                    </div>
                </div>
                <div class="form-group">
                    <label>Ramadan Day *</label>
                    <input type="number" name="ramadan_day" id="edit_ramadan_day" min="1" max="30" required>
                </div>
                <div style="margin-top: 20px; display: flex; gap: 10px;">
                    <button type="submit" class="btn">Update</button>
                    <button type="button" onclick="closeEditModal()" class="btn" style="background: #666;">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <style>
        .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; margin-bottom: 5px; font-weight: 500; }
        .form-group input, .form-group select { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
        .btn-small {
            padding: 6px 12px;
            font-size: 12px;
        }
        .btn-danger {
            background: #c33;
            color: white;
        }
        .btn-danger:hover {
            background: #a22;
        }
        .btn-danger:disabled {
            background: #dc3545;
            opacity: 0.6;
            cursor: not-allowed;
        }
        .btn-danger:not(:disabled):hover {
            background: #c82333;
            opacity: 1;
        }
    </style>

    <script>
    function deleteEntry(id) {
        if (!confirm('Are you sure you want to delete this prayer time entry?')) return;
        var f = document.getElementById('singleDeleteForm');
        if (f) {
            document.getElementById('singleDeleteId').value = id;
            f.submit();
        }
    }

    function editEntry(entry) {
        document.getElementById('edit_id').value = entry.id;
        document.getElementById('edit_location_id').value = entry.location_id;
        document.getElementById('edit_date').value = entry.date;
        document.getElementById('edit_fajr').value = entry.fajr.substring(0, 5);
        document.getElementById('edit_sunrise').value = entry.sunrise.substring(0, 5);
        document.getElementById('edit_dhuhr').value = entry.dhuhr.substring(0, 5);
        document.getElementById('edit_asr').value = entry.asr.substring(0, 5);
        document.getElementById('edit_maghrib').value = entry.maghrib.substring(0, 5);
        document.getElementById('edit_isha').value = entry.isha.substring(0, 5);
        document.getElementById('edit_sehri_end').value = entry.sehri_end.substring(0, 5);
        document.getElementById('edit_iftar').value = entry.iftar.substring(0, 5);
        document.getElementById('edit_ramadan_day').value = entry.ramadan_day;
        document.getElementById('editModal').style.display = 'flex';
    }

    function closeEditModal() {
        document.getElementById('editModal').style.display = 'none';
    }

    // Close modal on outside click
    document.getElementById('editModal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeEditModal();
        }
    });

    // Bulk delete functions
    function toggleSelectAll(selectAllCheckbox) {
        const checkboxes = document.querySelectorAll('.row-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = selectAllCheckbox.checked;
        });
        updateBulkDeleteButton();
    }

    function updateBulkDeleteButton() {
        const checkboxes = document.querySelectorAll('.row-checkbox:checked');
        const count = checkboxes.length;
        const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
        const selectedCount = document.getElementById('selectedCount');
        
        if (selectedCount) {
            selectedCount.textContent = count;
        }
        
        if (bulkDeleteBtn) {
            bulkDeleteBtn.disabled = count === 0;
        }
        
        // Update select all checkbox state
        const selectAllCheckbox = document.getElementById('selectAll');
        if (selectAllCheckbox) {
            const allCheckboxes = document.querySelectorAll('.row-checkbox');
            const checkedCount = document.querySelectorAll('.row-checkbox:checked').length;
            selectAllCheckbox.checked = allCheckboxes.length > 0 && checkedCount === allCheckboxes.length;
            selectAllCheckbox.indeterminate = checkedCount > 0 && checkedCount < allCheckboxes.length;
        }
    }

    function confirmBulkDelete() {
        const checkboxes = document.querySelectorAll('.row-checkbox:checked');
        const count = checkboxes.length;
        
        if (count === 0) {
            alert('Please select at least one entry to delete.');
            return false;
        }
        
        return confirm(`Are you sure you want to delete ${count} selected entry/entries? This action cannot be undone.`);
    }

    // Initialize bulk delete button state on page load (only when bulk form exists)
    document.addEventListener('DOMContentLoaded', function() {
        if (document.getElementById('bulkDeleteForm')) {
            updateBulkDeleteButton();
        }
    });
    </script>
</body>
</html>
