<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';

// Restore flash message after redirect (POST-Redirect-GET)
if (isset($_SESSION['ramadan_schedule_message'])) {
    $message = $_SESSION['ramadan_schedule_message'];
    unset($_SESSION['ramadan_schedule_message']);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] === 'add') {
            // Add single Ramadan schedule entry
            $stmt = $pdo->prepare("
                INSERT INTO prayer_times 
                (location_id, date, fajr, sunrise, dhuhr, asr, maghrib, isha, sehri_end, iftar, ramadan_day)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    sehri_end = VALUES(sehri_end),
                    iftar = VALUES(iftar),
                    ramadan_day = VALUES(ramadan_day),
                    fajr = VALUES(fajr),
                    sunrise = VALUES(sunrise),
                    dhuhr = VALUES(dhuhr),
                    asr = VALUES(asr),
                    maghrib = VALUES(maghrib),
                    isha = VALUES(isha)
            ");
            
            try {
                $stmt->execute([
                    $_POST['location_id'],
                    $_POST['date'],
                    $_POST['fajr'] ?? '04:30:00',
                    $_POST['sunrise'] ?? '06:00:00',
                    $_POST['dhuhr'] ?? '12:15:00',
                    $_POST['asr'] ?? '15:30:00',
                    $_POST['maghrib'] ?? '18:00:00',
                    $_POST['isha'] ?? '19:30:00',
                    $_POST['sehri_end'],
                    $_POST['iftar'],
                    $_POST['ramadan_day'],
                ]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Ramadan schedule entry added successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['ramadan_schedule_message'] = $message;
            header('Location: ramadan-schedule.php?location_id=' . (int)($_POST['location_id'] ?? 0));
            exit;
        } elseif ($_POST['action'] === 'bulk_add') {
            // Bulk add Ramadan schedule for date range
            $locationId = (int)$_POST['location_id'];
            $startDate = $_POST['start_date'];
            $endDate = $_POST['end_date'];
            $startRamadanDay = (int)$_POST['start_ramadan_day'];
            
            $currentDate = new DateTime($startDate);
            $endDateTime = new DateTime($endDate);
            $ramadanDay = $startRamadanDay;
            $successCount = 0;
            $errorCount = 0;
            
            $stmt = $pdo->prepare("
                INSERT INTO prayer_times 
                (location_id, date, fajr, sunrise, dhuhr, asr, maghrib, isha, sehri_end, iftar, ramadan_day)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                    sehri_end = VALUES(sehri_end),
                    iftar = VALUES(iftar),
                    ramadan_day = VALUES(ramadan_day),
                    fajr = VALUES(fajr),
                    sunrise = VALUES(sunrise),
                    dhuhr = VALUES(dhuhr),
                    asr = VALUES(asr),
                    maghrib = VALUES(maghrib),
                    isha = VALUES(isha)
            ");
            
            while ($currentDate <= $endDateTime) {
                $dateStr = $currentDate->format('Y-m-d');
                $sehriEnd = $_POST['sehri_end_' . $dateStr] ?? $_POST['default_sehri_end'];
                $iftar = $_POST['iftar_' . $dateStr] ?? $_POST['default_iftar'];
                
                try {
                    $stmt->execute([
                        $locationId,
                        $dateStr,
                        $_POST['default_fajr'] ?? '04:30:00',
                        $_POST['default_sunrise'] ?? '06:00:00',
                        $_POST['default_dhuhr'] ?? '12:15:00',
                        $_POST['default_asr'] ?? '15:30:00',
                        $_POST['default_maghrib'] ?? '18:00:00',
                        $_POST['default_isha'] ?? '19:30:00',
                        $sehriEnd,
                        $iftar,
                        $ramadanDay,
                    ]);
                    $successCount++;
                } catch (Exception $e) {
                    $errorCount++;
                }
                
                $currentDate->modify('+1 day');
                $ramadanDay++;
            }
            
            if ($errorCount > 0) {
                $message = '<div style="background: #ffd; color: #660; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Bulk add completed: ' . $successCount . ' entries added, ' . $errorCount . ' errors.</div>';
            } else {
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Successfully added ' . $successCount . ' Ramadan schedule entries!</div>';
            }
            $_SESSION['ramadan_schedule_message'] = $message;
            header('Location: ramadan-schedule.php?location_id=' . (int)$locationId);
            exit;
        } elseif ($_POST['action'] === 'update') {
            // Update existing entry
            $stmt = $pdo->prepare("
                UPDATE prayer_times 
                SET sehri_end = ?, iftar = ?, ramadan_day = ?
                WHERE id = ?
            ");
            
            try {
                $stmt->execute([
                    $_POST['sehri_end'],
                    $_POST['iftar'],
                    $_POST['ramadan_day'],
                    $_POST['id'],
                ]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Ramadan schedule updated successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['ramadan_schedule_message'] = $message;
            header('Location: ramadan-schedule.php?location_id=' . (int)($_POST['location_id'] ?? 0));
            exit;
        } elseif ($_POST['action'] === 'delete') {
            // Delete entry
            $stmt = $pdo->prepare("DELETE FROM prayer_times WHERE id = ?");
            try {
                $stmt->execute([$_POST['id']]);
                $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Ramadan schedule entry deleted successfully!</div>';
            } catch (Exception $e) {
                $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
            }
            $_SESSION['ramadan_schedule_message'] = $message;
            header('Location: ramadan-schedule.php?location_id=' . (int)($_POST['location_id'] ?? 0));
            exit;
        } elseif ($_POST['action'] === 'bulk_delete') {
            // Bulk delete entries
            if (isset($_POST['selected_ids']) && is_array($_POST['selected_ids']) && !empty($_POST['selected_ids'])) {
                $ids = array_map('intval', $_POST['selected_ids']);
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $stmt = $pdo->prepare("DELETE FROM prayer_times WHERE id IN ($placeholders)");
                
                try {
                    $stmt->execute($ids);
                    $deletedCount = $stmt->rowCount();
                    $message = '<div style="background: #dfd; color: #060; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Successfully deleted ' . $deletedCount . ' Ramadan schedule entry/entries!</div>';
                } catch (Exception $e) {
                    $message = '<div style="background: #fdd; color: #600; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
                }
            } else {
                $message = '<div style="background: #ffd; color: #660; padding: 10px; border-radius: 6px; margin-bottom: 20px;">Please select at least one entry to delete.</div>';
            }
            $_SESSION['ramadan_schedule_message'] = $message;
            header('Location: ramadan-schedule.php?location_id=' . (int)($_POST['location_id'] ?? 0));
            exit;
        }
    }
}

// Get selected location
$selectedLocationId = isset($_GET['location_id']) ? (int)$_GET['location_id'] : null;

// Get locations
$locations = $pdo->query("SELECT id, name, district FROM locations WHERE is_active = 1 ORDER BY district, name")->fetchAll();

// Get Ramadan schedule for selected location
$ramadanSchedule = [];
$scheduleDates = []; // For calendar view
if ($selectedLocationId) {
    $stmt = $pdo->prepare("
        SELECT pt.*, l.name as location_name, l.district
        FROM prayer_times pt
        JOIN locations l ON pt.location_id = l.id
        WHERE pt.location_id = ? AND pt.ramadan_day > 0
        ORDER BY pt.date ASC
    ");
    $stmt->execute([$selectedLocationId]);
    $ramadanSchedule = $stmt->fetchAll();
    
    // Create date map for calendar
    foreach ($ramadanSchedule as $entry) {
        $scheduleDates[$entry['date']] = [
            'day' => $entry['ramadan_day'],
            'sehri' => substr($entry['sehri_end'], 0, 5),
            'iftar' => substr($entry['iftar'], 0, 5),
        ];
    }
}

// Get current month/year for calendar view
$viewMonth = isset($_GET['month']) ? (int)$_GET['month'] : date('n');
$viewYear = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');
?>
<?php include 'header.php'; ?>

<div class="container">
    <h2>Manage Ramadan Schedule</h2>
    
    <?= $message ?>
    
    <!-- Location Selector -->
    <div class="section">
        <h3>Select Location</h3>
        <form method="GET" style="max-width: 400px;">
            <div class="form-group">
                <label>Location *</label>
                <select name="location_id" required style="width: 100%; padding: 8px;" onchange="this.form.submit()">
                    <option value="">-- Select Location --</option>
                    <?php foreach ($locations as $loc): ?>
                        <option value="<?= $loc['id'] ?>" <?= $selectedLocationId == $loc['id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($loc['name'] . ', ' . $loc['district']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </form>
    </div>
    
    <?php if ($selectedLocationId): ?>
        <!-- Calendar View -->
        <div class="section">
            <h3>Ramadan Calendar View - <?= htmlspecialchars($ramadanSchedule[0]['location_name'] ?? 'Selected Location') ?></h3>
            <div style="margin-bottom: 20px;">
                <a href="?location_id=<?= $selectedLocationId ?>&month=<?= $viewMonth == 1 ? 12 : $viewMonth - 1 ?>&year=<?= $viewMonth == 1 ? $viewYear - 1 : $viewYear ?>" class="btn" style="margin-right: 10px;">← Previous Month</a>
                <strong style="margin: 0 20px;"><?= date('F Y', mktime(0, 0, 0, $viewMonth, 1, $viewYear)) ?></strong>
                <a href="?location_id=<?= $selectedLocationId ?>&month=<?= $viewMonth == 12 ? 1 : $viewMonth + 1 ?>&year=<?= $viewMonth == 12 ? $viewYear + 1 : $viewYear ?>" class="btn">Next Month →</a>
            </div>
            <?php
            $firstDay = mktime(0, 0, 0, $viewMonth, 1, $viewYear);
            $daysInMonth = date('t', $firstDay);
            $dayOfWeek = date('w', $firstDay);
            $dayOfWeek = $dayOfWeek == 0 ? 6 : $dayOfWeek - 1; // Monday = 0
            ?>
            <div class="calendar-container">
                <div class="calendar-header">
                    <div>Mon</div>
                    <div>Tue</div>
                    <div>Wed</div>
                    <div>Thu</div>
                    <div>Fri</div>
                    <div>Sat</div>
                    <div>Sun</div>
                </div>
                <div class="calendar-grid">
                    <?php
                    // Empty cells for days before month starts
                    for ($i = 0; $i < $dayOfWeek; $i++) {
                        echo '<div class="calendar-day empty"></div>';
                    }
                    
                    // Days of the month
                    for ($day = 1; $day <= $daysInMonth; $day++) {
                        $dateStr = sprintf('%04d-%02d-%02d', $viewYear, $viewMonth, $day);
                        $hasData = isset($scheduleDates[$dateStr]);
                        $isToday = $dateStr == date('Y-m-d');
                        $class = 'calendar-day';
                        if ($hasData) $class .= ' has-data';
                        if ($isToday) $class .= ' today';
                        
                        echo '<div class="' . $class . '" onclick="selectDate(\'' . $dateStr . '\')" title="' . ($hasData ? 'Day ' . $scheduleDates[$dateStr]['day'] . ': Sehri ' . $scheduleDates[$dateStr]['sehri'] . ', Iftar ' . $scheduleDates[$dateStr]['iftar'] : 'Click to add') . '">';
                        echo '<div class="day-number">' . $day . '</div>';
                        if ($hasData) {
                            echo '<div class="day-info">';
                            echo '<div class="ramadan-day">Day ' . $scheduleDates[$dateStr]['day'] . '</div>';
                            echo '<div class="times">' . $scheduleDates[$dateStr]['sehri'] . ' / ' . $scheduleDates[$dateStr]['iftar'] . '</div>';
                            echo '</div>';
                        }
                        if ($isToday) {
                            echo '<div class="today-badge">Today</div>';
                        }
                        echo '</div>';
                    }
                    ?>
                </div>
            </div>
            <div style="margin-top: 20px; padding: 15px; background: #f5f5f5; border-radius: 8px;">
                <strong>Legend:</strong>
                <span style="display: inline-block; margin-left: 20px;">
                    <span style="display: inline-block; width: 20px; height: 20px; background: #4CAF50; border-radius: 4px; vertical-align: middle; margin-right: 5px;"></span>
                    Has Ramadan Schedule
                </span>
                <span style="display: inline-block; margin-left: 20px;">
                    <span style="display: inline-block; width: 20px; height: 20px; background: #2196F3; border-radius: 4px; vertical-align: middle; margin-right: 5px;"></span>
                    Today
                </span>
                <p style="margin-top: 10px; color: #666; font-size: 14px;">
                    <strong>Tip:</strong> Click on any date in the calendar to quickly add or edit Ramadan schedule for that date.
                </p>
            </div>
        </div>
        
        <!-- Add Single Entry -->
        <div class="section">
            <h3>Add/Edit Single Ramadan Entry</h3>
            <form method="POST" style="max-width: 600px;">
                <input type="hidden" name="action" value="add">
                <input type="hidden" name="location_id" value="<?= $selectedLocationId ?>">
                <div class="form-row">
                    <div class="form-group">
                        <label>Date * (YYYY-MM-DD)</label>
                        <input type="date" name="date" id="entry_date" required>
                        <small style="color: #666; font-size: 12px;">Or click a date in the calendar above</small>
                    </div>
                    <div class="form-group">
                        <label>Ramadan Day * (1-30)</label>
                        <input type="number" name="ramadan_day" min="1" max="30" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Sehri End * (HH:MM)</label>
                        <input type="time" name="sehri_end" required>
                    </div>
                    <div class="form-group">
                        <label>Iftar * (HH:MM)</label>
                        <input type="time" name="iftar" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Fajr (HH:MM)</label>
                        <input type="time" name="fajr" value="04:30">
                    </div>
                    <div class="form-group">
                        <label>Sunrise (HH:MM)</label>
                        <input type="time" name="sunrise" value="06:00">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Dhuhr (HH:MM)</label>
                        <input type="time" name="dhuhr" value="12:15">
                    </div>
                    <div class="form-group">
                        <label>Asr (HH:MM)</label>
                        <input type="time" name="asr" value="15:30">
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Maghrib (HH:MM)</label>
                        <input type="time" name="maghrib" value="18:00">
                    </div>
                    <div class="form-group">
                        <label>Isha (HH:MM)</label>
                        <input type="time" name="isha" value="19:30">
                    </div>
                </div>
                <button type="submit" class="btn">Add Entry</button>
            </form>
        </div>
        
        <!-- Quick Actions -->
        <div class="section">
            <h3>Quick Actions</h3>
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px;">
                <button onclick="setRamadan2026()" class="btn" style="width: 100%;">Set Full Ramadan 2026 (Feb 18 - Mar 19)</button>
                <button onclick="setRamadan2027()" class="btn" style="width: 100%;">Set Full Ramadan 2027 (Feb 7 - Mar 9)</button>
                <button onclick="setCurrentMonth()" class="btn" style="width: 100%;">Set Current Month</button>
            </div>
        </div>
        
        <!-- Bulk Add -->
        <div class="section">
            <h3>Bulk Add Ramadan Schedule (Date Range)</h3>
            <form method="POST" id="bulkForm" style="max-width: 600px;">
                <input type="hidden" name="action" value="bulk_add">
                <input type="hidden" name="location_id" value="<?= $selectedLocationId ?>">
                <div class="form-row">
                    <div class="form-group">
                        <label>Start Date *</label>
                        <input type="date" name="start_date" id="start_date" required>
                    </div>
                    <div class="form-group">
                        <label>End Date *</label>
                        <input type="date" name="end_date" id="end_date" required>
                    </div>
                </div>
                <div class="form-group">
                    <label>Start Ramadan Day * (e.g., 1 for first day of Ramadan)</label>
                    <input type="number" name="start_ramadan_day" min="1" max="30" value="1" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label>Default Sehri End (HH:MM)</label>
                        <input type="time" name="default_sehri_end" value="04:25" required>
                    </div>
                    <div class="form-group">
                        <label>Default Iftar (HH:MM)</label>
                        <input type="time" name="default_iftar" value="18:05" required>
                    </div>
                </div>
                <div class="info-box">
                    <strong>Note:</strong> This will create entries for all dates in the range. You can edit individual entries later.
                </div>
                <button type="submit" class="btn">Bulk Add Schedule</button>
            </form>
        </div>
        
        <!-- Test API -->
        <div class="section" style="background: #fff3cd; border-left: 4px solid #ffc107;">
            <h3>🔍 Test API & Debug</h3>
            <p style="margin-bottom: 15px;">Test if the API is returning data correctly for this location:</p>
            <a href="../api/ramadan-schedule.php?location_id=<?= $selectedLocationId ?>" target="_blank" class="btn" style="background: #ffc107; color: #000;">
                Test API Response
            </a>
            <p style="margin-top: 15px; font-size: 14px; color: #856404;">
                <strong>Note:</strong> This will open the API response in a new tab. You should see JSON data if entries exist.
            </p>
            <?php
            // Check database directly
            $dbCheck = $pdo->prepare("
                SELECT COUNT(*) as total,
                       COUNT(CASE WHEN ramadan_day > 0 THEN 1 END) as ramadan_entries,
                       COUNT(CASE WHEN ramadan_day = 0 OR ramadan_day IS NULL THEN 1 END) as non_ramadan_entries
                FROM prayer_times
                WHERE location_id = ?
            ");
            $dbCheck->execute([$selectedLocationId]);
            $dbStats = $dbCheck->fetch();
            ?>
            <div style="margin-top: 15px; padding: 10px; background: white; border-radius: 4px;">
                <strong>Database Stats for this Location:</strong>
                <ul style="margin: 10px 0; padding-left: 20px;">
                    <li>Total prayer time entries: <?= $dbStats['total'] ?></li>
                    <li>Ramadan entries (ramadan_day > 0): <strong style="color: #28a745;"><?= $dbStats['ramadan_entries'] ?></strong></li>
                    <li>Non-Ramadan entries: <?= $dbStats['non_ramadan_entries'] ?></li>
                </ul>
                <?php if ($dbStats['ramadan_entries'] == 0 && $dbStats['total'] > 0): ?>
                    <p style="color: #dc3545; margin-top: 10px;">
                        ⚠️ <strong>Issue Found:</strong> You have <?= $dbStats['total'] ?> prayer time entries, but none have ramadan_day > 0. 
                        Make sure when adding entries, you set the "Ramadan Day" field to 1-30.
                    </p>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- Existing Schedule -->
        <div class="section">
            <h3>Ramadan Schedule for <?= htmlspecialchars($ramadanSchedule[0]['location_name'] ?? 'Selected Location') ?></h3>
            <?php if (empty($ramadanSchedule)): ?>
                <p style="color: #666; padding: 20px; text-align: center;">No Ramadan schedule entries found for this location. Add entries using the forms above.</p>
            <?php else: ?>
                <!-- Hidden form for single delete (used via JS to avoid nested forms) -->
                <form method="POST" id="singleDeleteForm" style="display: none;">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="location_id" value="<?= $selectedLocationId ?>">
                    <input type="hidden" name="id" id="singleDeleteId">
                </form>

                <!-- Bulk Delete Form -->
                <form method="POST" id="bulkDeleteForm" onsubmit="return confirmBulkDelete();">
                    <input type="hidden" name="action" value="bulk_delete">
                    <input type="hidden" name="location_id" value="<?= $selectedLocationId ?>">
                    <div style="margin-bottom: 15px;">
                        <button type="submit" id="bulkDeleteBtn" class="btn btn-danger" disabled style="background: #dc3545; color: white; padding: 10px 20px; border: none; border-radius: 6px; cursor: not-allowed; font-weight: 600; opacity: 0.6;">
                            Delete Selected (<span id="selectedCount">0</span>)
                        </button>
                    </div>
                    
                    <div style="overflow-x: auto;">
                        <table>
                            <thead>
                                <tr>
                                    <th style="width: 50px;">
                                        <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                                    </th>
                                    <th>Ramadan Day</th>
                                    <th>Date</th>
                                    <th>Sehri End</th>
                                    <th>Iftar</th>
                                    <th>Fajr</th>
                                    <th>Maghrib</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($ramadanSchedule as $entry): ?>
                                    <tr>
                                        <td style="text-align: center;">
                                            <input type="checkbox" name="selected_ids[]" value="<?= $entry['id'] ?>" class="row-checkbox" onchange="updateBulkDeleteButton()">
                                        </td>
                                        <td><strong><?= htmlspecialchars($entry['ramadan_day']) ?></strong></td>
                                        <td><?= htmlspecialchars($entry['date']) ?></td>
                                        <td><?= htmlspecialchars(substr($entry['sehri_end'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($entry['iftar'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($entry['fajr'], 0, 5)) ?></td>
                                        <td><?= htmlspecialchars(substr($entry['maghrib'], 0, 5)) ?></td>
                                        <td>
                                            <button type="button" onclick="editEntry(<?= htmlspecialchars(json_encode($entry)) ?>)" class="btn-small" style="margin-right: 5px;">Edit</button>
                                            <button type="button" class="btn-small btn-danger" onclick="deleteEntry(<?= (int)$entry['id'] ?>)">Delete</button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<!-- Edit Modal -->
<div id="editModal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000; align-items: center; justify-content: center;">
    <div style="background: white; padding: 30px; border-radius: 12px; max-width: 500px; width: 90%; max-height: 90vh; overflow-y: auto;">
        <h3 style="margin-bottom: 20px;">Edit Ramadan Schedule Entry</h3>
        <form method="POST" id="editForm">
            <input type="hidden" name="action" value="update">
            <input type="hidden" name="location_id" value="<?= $selectedLocationId ?? '' ?>">
            <input type="hidden" name="id" id="edit_id">
            <div class="form-group">
                <label>Date</label>
                <input type="date" name="date" id="edit_date" readonly style="background: #f5f5f5;">
            </div>
            <div class="form-group">
                <label>Ramadan Day *</label>
                <input type="number" name="ramadan_day" id="edit_ramadan_day" min="1" max="30" required>
            </div>
            <div class="form-row">
                <div class="form-group">
                    <label>Sehri End *</label>
                    <input type="time" name="sehri_end" id="edit_sehri_end" required>
                </div>
                <div class="form-group">
                    <label>Iftar *</label>
                    <input type="time" name="iftar" id="edit_iftar" required>
                </div>
            </div>
            <div style="margin-top: 20px; display: flex; gap: 10px;">
                <button type="submit" class="btn">Update</button>
                <button type="button" onclick="closeEditModal()" class="btn" style="background: #666;">Cancel</button>
            </div>
        </form>
    </div>
</div>

<style>
    .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
    .form-group { margin-bottom: 15px; }
    .form-group label { display: block; margin-bottom: 5px; font-weight: 500; }
    .form-group input, .form-group select { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; }
    .info-box {
        background: #e3f2fd;
        border-left: 4px solid #2196F3;
        padding: 12px;
        margin: 15px 0;
        border-radius: 4px;
    }
    .btn-small {
        padding: 6px 12px;
        font-size: 12px;
    }
    .btn-danger {
        background: #c33;
    }
    .btn-danger:hover {
        background: #a22;
    }
    .btn-danger:disabled {
        background: #dc3545;
        opacity: 0.6;
        cursor: not-allowed;
    }
    .btn-danger:not(:disabled):hover {
        background: #c82333;
        opacity: 1;
    }
    .calendar-container {
        background: white;
        border: 1px solid #ddd;
        border-radius: 8px;
        overflow: hidden;
    }
    .calendar-header {
        display: grid;
        grid-template-columns: repeat(7, 1fr);
        background: #2D5016;
        color: white;
        font-weight: 600;
    }
    .calendar-header div {
        padding: 12px;
        text-align: center;
        border-right: 1px solid rgba(255,255,255,0.2);
    }
    .calendar-header div:last-child {
        border-right: none;
    }
    .calendar-grid {
        display: grid;
        grid-template-columns: repeat(7, 1fr);
    }
    .calendar-day {
        min-height: 100px;
        border-right: 1px solid #e0e0e0;
        border-bottom: 1px solid #e0e0e0;
        padding: 8px;
        cursor: pointer;
        transition: background 0.2s;
        position: relative;
    }
    .calendar-day:nth-child(7n) {
        border-right: none;
    }
    .calendar-day:hover {
        background: #f5f5f5;
    }
    .calendar-day.empty {
        background: #fafafa;
        cursor: default;
    }
    .calendar-day.has-data {
        background: #e8f5e9;
        border-left: 4px solid #4CAF50;
    }
    .calendar-day.today {
        background: #e3f2fd;
        border: 2px solid #2196F3;
    }
    .calendar-day.today.has-data {
        background: #c8e6c9;
    }
    .day-number {
        font-weight: bold;
        font-size: 16px;
        margin-bottom: 4px;
    }
    .day-info {
        font-size: 11px;
        color: #2D5016;
    }
    .ramadan-day {
        font-weight: 600;
        margin-bottom: 2px;
    }
    .times {
        color: #666;
    }
    .today-badge {
        position: absolute;
        top: 4px;
        right: 4px;
        background: #2196F3;
        color: white;
        font-size: 9px;
        padding: 2px 6px;
        border-radius: 10px;
        font-weight: bold;
    }
</style>

<script>
function deleteEntry(id) {
    if (!confirm('Are you sure you want to delete this entry?')) return;
    var f = document.getElementById('singleDeleteForm');
    if (f) {
        document.getElementById('singleDeleteId').value = id;
        f.submit();
    }
}

function editEntry(entry) {
    document.getElementById('edit_id').value = entry.id;
    document.getElementById('edit_date').value = entry.date;
    document.getElementById('edit_ramadan_day').value = entry.ramadan_day;
    document.getElementById('edit_sehri_end').value = entry.sehri_end.substring(0, 5);
    document.getElementById('edit_iftar').value = entry.iftar.substring(0, 5);
    document.getElementById('editModal').style.display = 'flex';
}

function closeEditModal() {
    document.getElementById('editModal').style.display = 'none';
}

// Close modal on outside click
document.getElementById('editModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeEditModal();
    }
});

// Select date from calendar
function selectDate(dateStr) {
    const dateInput = document.getElementById('entry_date');
    if (dateInput) {
        dateInput.value = dateStr;
        dateInput.scrollIntoView({ behavior: 'smooth', block: 'center' });
        // Highlight the form
        const form = dateInput.closest('form');
        if (form) {
            form.style.border = '2px solid #4CAF50';
            form.style.borderRadius = '8px';
            form.style.padding = '15px';
            setTimeout(() => {
                form.style.border = '';
                form.style.borderRadius = '';
                form.style.padding = '';
            }, 2000);
        }
    }
}

// Quick action functions
function setRamadan2026() {
    document.getElementById('start_date').value = '2026-02-18';
    document.getElementById('end_date').value = '2026-03-19';
    document.querySelector('input[name="start_ramadan_day"]').value = 1;
    document.getElementById('bulkForm').scrollIntoView({ behavior: 'smooth' });
}

function setRamadan2027() {
    document.getElementById('start_date').value = '2027-02-07';
    document.getElementById('end_date').value = '2027-03-09';
    document.querySelector('input[name="start_ramadan_day"]').value = 1;
    document.getElementById('bulkForm').scrollIntoView({ behavior: 'smooth' });
}

function setCurrentMonth() {
    const now = new Date();
    const year = now.getFullYear();
    const month = now.getMonth() + 1;
    const firstDay = new Date(year, month - 1, 1);
    const lastDay = new Date(year, month, 0);
    
    document.getElementById('start_date').value = firstDay.toISOString().split('T')[0];
    document.getElementById('end_date').value = lastDay.toISOString().split('T')[0];
    document.querySelector('input[name="start_ramadan_day"]').value = 1;
    document.getElementById('bulkForm').scrollIntoView({ behavior: 'smooth' });
}

// Bulk delete functions
function toggleSelectAll(selectAllCheckbox) {
    const checkboxes = document.querySelectorAll('.row-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = selectAllCheckbox.checked;
    });
    updateBulkDeleteButton();
}

function updateBulkDeleteButton() {
    const checkboxes = document.querySelectorAll('.row-checkbox:checked');
    const count = checkboxes.length;
    const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
    const selectedCount = document.getElementById('selectedCount');
    
    if (selectedCount) {
        selectedCount.textContent = count;
    }
    
    if (bulkDeleteBtn) {
        bulkDeleteBtn.disabled = count === 0;
    }
    
    // Update select all checkbox state
    const selectAllCheckbox = document.getElementById('selectAll');
    if (selectAllCheckbox) {
        const allCheckboxes = document.querySelectorAll('.row-checkbox');
        const checkedCount = document.querySelectorAll('.row-checkbox:checked').length;
        selectAllCheckbox.checked = allCheckboxes.length > 0 && checkedCount === allCheckboxes.length;
        selectAllCheckbox.indeterminate = checkedCount > 0 && checkedCount < allCheckboxes.length;
    }
}

function confirmBulkDelete() {
    const checkboxes = document.querySelectorAll('.row-checkbox:checked');
    const count = checkboxes.length;
    
    if (count === 0) {
        alert('Please select at least one entry to delete.');
        return false;
    }
    
    return confirm(`Are you sure you want to delete ${count} selected entry/entries? This action cannot be undone.`);
}

// Initialize bulk delete button state on page load (only when bulk form exists)
document.addEventListener('DOMContentLoaded', function() {
    if (document.getElementById('bulkDeleteForm')) {
        updateBulkDeleteButton();
    }
});
</script>

</body>
</html>
