<?php
session_start();
require_once __DIR__ . '/../config/database.php';

if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

$pdo = getDBConnection();
$message = '';
$editItem = null;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action']) && $_POST['action'] === 'add') {
        $stmt = $pdo->prepare("
            INSERT INTO surahs 
            (surah_name, surah_number, audio_url, description, display_order, is_active)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        try {
            $stmt->execute([
                $_POST['surah_name'],
                $_POST['surah_number'],
                $_POST['audio_url'],
                $_POST['description'] ?? null,
                $_POST['display_order'] ?? 0,
                isset($_POST['is_active']) ? 1 : 0,
            ]);
            $message = '<div class="alert alert-success">Surah added successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    elseif (isset($_POST['action']) && $_POST['action'] === 'edit') {
        $stmt = $pdo->prepare("
            UPDATE surahs 
            SET surah_name = ?, surah_number = ?, audio_url = ?, description = ?, display_order = ?, is_active = ?
            WHERE id = ?
        ");
        
        try {
            $stmt->execute([
                $_POST['surah_name'],
                $_POST['surah_number'],
                $_POST['audio_url'],
                $_POST['description'] ?? null,
                $_POST['display_order'] ?? 0,
                isset($_POST['is_active']) ? 1 : 0,
                $_POST['id']
            ]);
            $message = '<div class="alert alert-success">Surah updated successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    elseif (isset($_POST['action']) && $_POST['action'] === 'delete') {
        $stmt = $pdo->prepare("DELETE FROM surahs WHERE id = ?");
        try {
            $stmt->execute([$_POST['id']]);
            $message = '<div class="alert alert-success">Surah deleted successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-error">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// Handle edit request
if (isset($_GET['edit'])) {
    $stmt = $pdo->prepare("SELECT * FROM surahs WHERE id = ?");
    $stmt->execute([$_GET['edit']]);
    $editItem = $stmt->fetch();
}

// Get all surahs
$surahs = $pdo->query("SELECT * FROM surahs ORDER BY display_order, id")->fetchAll();
?>
<?php include 'header.php'; ?>

<div class="container">
    <h2>Manage Surahs (القرآن الكريم)</h2>
    
    <?= $message ?>
    
    <div class="section">
        <h3><?= $editItem ? 'Edit Surah' : 'Add New Surah' ?></h3>
        <?php if ($editItem): ?>
            <a href="surahs.php" class="btn btn-secondary" style="margin-bottom: 15px; display: inline-block;">← Cancel Edit</a>
        <?php endif; ?>
        <form method="POST" style="max-width: 800px;">
            <input type="hidden" name="action" value="<?= $editItem ? 'edit' : 'add' ?>">
            <?php if ($editItem): ?>
                <input type="hidden" name="id" value="<?= $editItem['id'] ?>">
            <?php endif; ?>
            
            <div class="form-group">
                <label>Surah Name (বাংলা) *</label>
                <input type="text" name="surah_name" value="<?= $editItem ? htmlspecialchars($editItem['surah_name']) : '' ?>" required placeholder="e.g., আল-ফাতিহা">
            </div>
            
            <div class="form-group">
                <label>Surah Number *</label>
                <input type="text" name="surah_number" value="<?= $editItem ? htmlspecialchars($editItem['surah_number']) : '' ?>" required placeholder="e.g., 1 (or Roman numerals)">
            </div>
            
            <div class="form-group">
                <label>Audio URL (MP3 Link) *</label>
                <input type="url" name="audio_url" value="<?= $editItem ? htmlspecialchars($editItem['audio_url']) : '' ?>" required placeholder="https://example.com/surah.mp3">
                <small style="color: #666; margin-top: 5px; display: block;">Must be a valid MP3 file URL. Make sure the server allows CORS.</small>
            </div>
            
            <div class="form-group">
                <label>Description (Optional)</label>
                <textarea name="description" rows="3" placeholder="Add any notes or description about this Surah..."><?= $editItem ? htmlspecialchars($editItem['description']) : '' ?></textarea>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Display Order</label>
                    <input type="number" name="display_order" value="<?= $editItem ? $editItem['display_order'] : '0' ?>" min="0">
                </div>
                <div class="form-group">
                    <label>
                        <input type="checkbox" name="is_active" <?= (!$editItem || $editItem['is_active']) ? 'checked' : '' ?>> Active
                    </label>
                </div>
            </div>
            
            <button type="submit" class="btn"><?= $editItem ? 'Update Surah' : 'Add Surah' ?></button>
        </form>
    </div>
    
    <div class="section">
        <h3>All Surahs (<?= count($surahs) ?>)</h3>
        
        <?php if (empty($surahs)): ?>
            <p style="color: #666;">No surahs added yet. <a href="#">Add one now</a>.</p>
        <?php else: ?>
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Surah Name</th>
                            <th>Number</th>
                            <th>Audio Link</th>
                            <th>Description</th>
                            <th>Order</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($surahs as $surah): ?>
                            <tr>
                                <td><?= $surah['id'] ?></td>
                                <td><strong><?= htmlspecialchars($surah['surah_name']) ?></strong></td>
                                <td><?= htmlspecialchars($surah['surah_number']) ?></td>
                                <td>
                                    <a href="<?= htmlspecialchars($surah['audio_url']) ?>" target="_blank" title="<?= htmlspecialchars($surah['audio_url']) ?>" style="color: #007bff; text-decoration: none;">
                                        🔗 Listen
                                    </a>
                                </td>
                                <td style="max-width: 250px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                    <?= htmlspecialchars(substr($surah['description'] ?? '', 0, 50)) ?><?= (strlen($surah['description'] ?? '') > 50) ? '...' : '' ?>
                                </td>
                                <td><?= $surah['display_order'] ?></td>
                                <td>
                                    <span class="badge <?= $surah['is_active'] ? 'badge-active' : 'badge-inactive' ?>">
                                        <?= $surah['is_active'] ? 'Active' : 'Inactive' ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <a href="?edit=<?= $surah['id'] ?>" class="btn btn-small btn-primary">Edit</a>
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to delete this Surah? This action cannot be undone.');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="<?= $surah['id'] ?>">
                                            <button type="submit" class="btn btn-small btn-danger">Delete</button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
    .form-row { 
        display: grid; 
        grid-template-columns: 1fr 1fr; 
        gap: 15px; 
    }
    .form-group { 
        margin-bottom: 15px; 
    }
    .form-group label { 
        display: block; 
        margin-bottom: 5px; 
        font-weight: 500; 
    }
    .form-group input, 
    .form-group select, 
    .form-group textarea { 
        width: 100%; 
        padding: 8px; 
        border: 1px solid #ddd; 
        border-radius: 4px; 
        font-size: 14px;
    }
    .form-group textarea { 
        font-family: inherit; 
        resize: vertical;
    }
    
    .alert { 
        padding: 12px 20px; 
        border-radius: 6px; 
        margin-bottom: 20px; 
    }
    .alert-success { 
        background: #d4edda; 
        color: #155724; 
        border: 1px solid #c3e6cb; 
    }
    .alert-error { 
        background: #f8d7da; 
        color: #721c24; 
        border: 1px solid #f5c6cb; 
    }
    
    .badge { 
        display: inline-block; 
        padding: 4px 8px; 
        border-radius: 4px; 
        font-size: 12px; 
        font-weight: 500; 
    }
    .badge-active { 
        background: #d4edda; 
        color: #155724; 
    }
    .badge-inactive { 
        background: #f8d7da; 
        color: #721c24; 
    }
    
    .action-buttons { 
        display: flex; 
        gap: 8px; 
        flex-wrap: wrap; 
    }
    .btn-small { 
        padding: 6px 12px; 
        font-size: 13px; 
    }
    .btn-primary { 
        background: #007bff; 
        color: white; 
        text-decoration: none;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        display: inline-block;
    }
    .btn-primary:hover { 
        background: #0056b3; 
    }
    .btn-danger { 
        background: #dc3545; 
        color: white;
        border: none;
        border-radius: 4px;
        cursor: pointer;
    }
    .btn-danger:hover { 
        background: #c82333; 
    }
    .btn-secondary { 
        background: #6c757d; 
        color: white;
        text-decoration: none;
        border: none;
        border-radius: 4px;
        cursor: pointer;
        display: inline-block;
    }
    .btn-secondary:hover { 
        background: #5a6268; 
    }
    
    table { 
        font-size: 14px; 
        width: 100%;
    }
    table td { 
        vertical-align: middle; 
    }
    
    small {
        font-size: 12px;
    }
</style>
</body>
</html>
