<?php
/**
 * API: Location-based Notifications
 * এই এন্ডপয়েন্ট ব্যবহার করে location-wise নোটিফিকেশন পাঠানো হয়
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config/database.php';

$pdo = getDBConnection();
$method = $_SERVER['REQUEST_METHOD'];
$request_uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);

try {
    // GET: সব সক্রিয় নোটিফিকেশন পান location দ্বারা ফিল্টার করে
    if ($method === 'GET' && preg_match('/\/api\/notifications/', $request_uri)) {
        $location_id = isset($_GET['location_id']) ? intval($_GET['location_id']) : null;
        
        if (!$location_id) {
            http_response_code(400);
            echo json_encode(['error' => 'location_id required']);
            exit;
        }
        
        // এই লোকেশনের জন্য প্রাসঙ্গিক নোটিফিকেশন খুঁজে বের করুন
        // ✅ Strict location filtering - show ONLY relevant notifications
        $stmt = $pdo->prepare("
            SELECT id, title, message, image_url, action_url, audio_url, created_at, 
                   is_scheduled, scheduled_time, targeting_type, location_ids
            FROM notifications
            WHERE is_active = 1 AND (
                targeting_type = 'all_locations' OR 
                (targeting_type = 'selected_locations' AND JSON_CONTAINS(COALESCE(location_ids, '[]'), ?))
            )
            ORDER BY created_at DESC
            LIMIT 50
        ");
        
        $location_json = json_encode($location_id);
        $stmt->execute([$location_json]);
        $notifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        http_response_code(200);
        echo json_encode(['success' => true, 'data' => $notifications]);
        exit;
    }
    
    // GET: সমস্ত লোকেশনের জন্য location-wise নোটিফিকেশন সংখ্যা পান
    if ($method === 'GET' && preg_match('/\/api\/notifications\/count-by-location/', $request_uri)) {
        $stmt = $pdo->prepare("
            SELECT l.id, l.name, l.district, 
                SUM(CASE WHEN n.is_active = 1 AND (
                    n.targeting_type = 'all_locations' OR 
                    JSON_CONTAINS(n.location_ids, JSON_QUOTE(l.id))
                ) THEN 1 ELSE 0 END) as notification_count
            FROM locations l
            LEFT JOIN notifications n ON 1=1
            WHERE l.is_active = 1
            GROUP BY l.id
            ORDER BY l.district, l.name
        ");
        
        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        http_response_code(200);
        echo json_encode(['success' => true, 'data' => $result]);
        exit;
    }
    
    // GET: নির্দিষ্ট নোটিফিকেশনের টার্গেট লোকেশন সম্পর্কে তথ্য পান
    if ($method === 'GET' && preg_match('/\/api\/notifications\/(\d+)\/targets/', $request_uri, $matches)) {
        $notification_id = intval($matches[1]);
        
        $stmt = $pdo->prepare("
            SELECT id, title, targeting_type, location_ids 
            FROM notifications 
            WHERE id = ?
        ");
        $stmt->execute([$notification_id]);
        $notification = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$notification) {
            http_response_code(404);
            echo json_encode(['error' => 'Notification not found']);
            exit;
        }
        
        $target_locations = [];
        if ($notification['targeting_type'] === 'all_locations') {
            // সব লোকেশন টার্গেট
            $loc_stmt = $pdo->query("SELECT id, name, district FROM locations WHERE is_active = 1 ORDER BY district");
            $target_locations = $loc_stmt->fetchAll(PDO::FETCH_ASSOC);
        } else if ($notification['targeting_type'] === 'selected_locations' && !empty($notification['location_ids'])) {
            // নির্বাচিত লোকেশন টার্গেট
            try {
                $selected_ids = json_decode($notification['location_ids'], true);
                if (is_array($selected_ids) && !empty($selected_ids)) {
                    $placeholders = implode(',', array_fill(0, count($selected_ids), '?'));
                    $loc_stmt = $pdo->prepare("SELECT id, name, district FROM locations WHERE id IN ($placeholders) ORDER BY district");
                    $loc_stmt->execute($selected_ids);
                    $target_locations = $loc_stmt->fetchAll(PDO::FETCH_ASSOC);
                }
            } catch (Exception $e) {
                // Error handling for JSON decode
            }
        }
        
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'notification_id' => $notification_id,
            'targeting_type' => $notification['targeting_type'],
            'target_locations' => $target_locations
        ]);
        exit;
    }
    
    // Invalid endpoint
    http_response_code(404);
    echo json_encode(['error' => 'Invalid endpoint']);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}
?>
