<?php
/**
 * নোটিফিকেশন বাল্ক আপলোড API এন্ডপয়েন্ট
 * POST /api/notifications-bulk-upload.php
 * 
 * মাল্টিপার্ট ফর্ম ডেটা:
 * - csv_file: CSV ফাইল
 * - validate_only: true (শুধু validation করবে, ডাটাবেসে insert করবে না)
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/response.php';
require_once __DIR__ . '/../includes/cors.php';

try {
    // এডমিন চেক করুন (অ্যাডভান্সড সিকিউরিটি - অপশনাল)
    // For now, allow anyone to validate. In production, add admin authentication.
    
    $pdo = getDBConnection();
    
    // CSV ফাইল চেক করুন
    if (!isset($_FILES['csv_file'])) {
        sendError('CSV ফাইল নেই', 400);
    }
    
    $file = $_FILES['csv_file'];
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        sendError('ফাইল আপলোড ত্রুটি: ' . $file['error'], 400);
    }
    
    if (pathinfo($file['name'], PATHINFO_EXTENSION) !== 'csv') {
        sendError('শুধুমাত্র CSV ফাইল আপলোড করুন', 400);
    }
    
    // সব লোকেশন লোড করুন (validation এর জন্য)
    $locations_stmt = $pdo->query("SELECT id FROM locations WHERE is_active = 1");
    $location_ids = array_column($locations_stmt->fetchAll(), 'id');
    $location_id_set = array_flip($location_ids);
    
    // CSV পড়ুন এবং ভ্যালিডেট করুন
    $file_handle = fopen($file['tmp_name'], 'r');
    
    // হেডার লাইন
    $header = fgetcsv($file_handle);
    
    if (!$header) {
        sendError('CSV ফাইল খালি বা ইনভ্যালিড', 400);
    }
    
    // প্রয়োজনীয় কলাম চেক করুন
    $required_columns = ['title', 'message', 'targeting_type'];
    $header_lower = array_map('strtolower', $header);
    $missing_columns = array_diff($required_columns, $header_lower);
    
    if (!empty($missing_columns)) {
        sendError('প্রয়োজনীয় কলাম নেই: ' . implode(', ', $missing_columns), 400);
    }
    
    // Column index ম্যাপ তৈরি করুন
    $column_map = array_flip($header_lower);
    
    // ডেটা রো ভ্যালিডেট করুন
    $results = [];
    $row_num = 2;
    $valid_count = 0;
    $invalid_count = 0;
    
    while (($row_data = fgetcsv($file_handle)) !== false) {
        if (empty(array_filter($row_data))) {
            continue; // খালি রো স্কিপ করুন
        }
        
        $title = isset($column_map['title']) ? trim($row_data[$column_map['title']]) : '';
        $message_text = isset($column_map['message']) ? trim($row_data[$column_map['message']]) : '';
        $targeting_type = isset($column_map['targeting_type']) ? trim($row_data[$column_map['targeting_type']]) : '';
        $location_ids_str = isset($column_map['location_ids']) ? trim($row_data[$column_map['location_ids']]) : '';
        $image_url = isset($column_map['image_url']) ? trim($row_data[$column_map['image_url']]) : '';
        $audio_url = isset($column_map['audio_url']) ? trim($row_data[$column_map['audio_url']]) : '';
        $action_url = isset($column_map['action_url']) ? trim($row_data[$column_map['action_url']]) : '';
        $is_scheduled = (isset($column_map['is_scheduled']) && strtolower(trim($row_data[$column_map['is_scheduled']])) === 'yes') ? 1 : 0;
        $scheduled_time = isset($column_map['scheduled_time']) ? trim($row_data[$column_map['scheduled_time']]) : '';
        $send_daily = (isset($column_map['send_daily']) && strtolower(trim($row_data[$column_map['send_daily']])) === 'yes') ? 1 : 0;
        $is_active = (isset($column_map['is_active']) && strtolower(trim($row_data[$column_map['is_active']])) === 'yes') ? 1 : 0;
        
        // ভ্যালিডেশন
        $errors = [];
        
        if (empty($title)) {
            $errors[] = 'টাইটেল খালি';
        } elseif (strlen($title) > 200) {
            $errors[] = 'টাইটেল 200 অক্ষরের বেশি';
        }
        
        if (empty($message_text)) {
            $errors[] = 'মেসেজ খালি';
        }
        
        if (!in_array($targeting_type, ['all_locations', 'selected_locations'], true)) {
            $errors[] = 'টার্গেটিং টাইপ অবৈধ (all_locations বা selected_locations হতে হবে)';
        }
        
        if ($targeting_type === 'selected_locations') {
            if (empty($location_ids_str)) {
                $errors[] = 'selected_locations নির্বাচিত হয়েছে কিন্তু location_ids খালি';
            } else {
                $id_array = array_filter(array_map('intval', array_map('trim', explode(',', $location_ids_str))));
                
                if (empty($id_array)) {
                    $errors[] = 'location_ids পার্স করা যায়নি';
                } else {
                    $invalid_ids = [];
                    foreach ($id_array as $loc_id) {
                        if (!isset($location_id_set[$loc_id])) {
                            $invalid_ids[] = $loc_id;
                        }
                    }
                    
                    if (!empty($invalid_ids)) {
                        $errors[] = 'অবৈধ location_ids: ' . implode(', ', $invalid_ids);
                    }
                }
            }
        }
        
        if ($is_scheduled && empty($scheduled_time)) {
            $errors[] = 'is_scheduled হয় কিন্তু scheduled_time খালি';
        }
        
        if ($is_scheduled && !empty($scheduled_time)) {
            if (!preg_match('/^\d{2}:\d{2}$/', $scheduled_time)) {
                $errors[] = 'scheduled_time ফরম্যাট অবৈধ (HH:MM ফরম্যাট প্রয়োজন)';
            }
        }
        
        if (!empty($image_url) && !filter_var($image_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'image_url অবৈধ URL';
        }
        
        if (!empty($audio_url) && !filter_var($audio_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'audio_url অবৈধ URL';
        }
        
        if (!empty($action_url) && !filter_var($action_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'action_url অবৈধ URL';
        }
        
        // ফলাফল রেকর্ড করুন
        if (empty($errors)) {
            $results[] = [
                'row' => $row_num,
                'title' => $title,
                'valid' => true
            ];
            $valid_count++;
        } else {
            $results[] = [
                'row' => $row_num,
                'title' => $title,
                'valid' => false,
                'errors' => $errors
            ];
            $invalid_count++;
        }
        
        $row_num++;
    }
    
    fclose($file_handle);
    
    // রেসপন্স পাঠান
    sendSuccess([
        'total_rows' => count($results),
        'valid_count' => $valid_count,
        'invalid_count' => $invalid_count,
        'rows' => $results
    ]);

} catch (Exception $e) {
    sendError('ত্রুটি: ' . $e->getMessage(), 500);
}

// রেসপন্স হেল্পার
function sendSuccess($data) {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'data' => $data
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

function sendError($message, $status_code = 400) {
    http_response_code($status_code);
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'error',
        'message' => $message
    ], JSON_UNESCAPED_UNICODE);
    exit;
}
?>
