<?php
/**
 * নোটিফিকেশন API এন্ডপয়েন্ট - FIXED VERSION with Location Filtering
 * GET /api/notifications.php - সমস্ত সক্রিয় নোটিফিকেশন পান
 * GET /api/notifications.php?location_id=1 - নির্দিষ্ট লোকেশনের নোটিফিকেশন পান
 * 
 * বর্তমান সময় অনুযায়ী সঠিক নোটিফিকেশন পায়:
 * 1. তাৎক্ষণিক নোটিফিকেশন (is_scheduled = 0) - যা কখনো পাঠানো হয়নি
 * 2. সময়-নির্ধারিত নোটিফিকেশন (is_scheduled = 1) - শুধুমাত্র যখন সময় মেলে
 * 3. Location filtering - যদি location_id পাঠানো হয়, শুধুমাত্র সেই লোকেশনের জন্য notifications
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/response.php';
require_once __DIR__ . '/../includes/cors.php';

try {
    $pdo = getDBConnection();
    
    date_default_timezone_set('Asia/Dhaka');
    $currentDate = date('Y-m-d');
    $currentTime = date('H:i'); // 04:30 format এ
    
    // 📍 Get location_id from query parameter if provided
    $location_id = isset($_GET['location_id']) ? intval($_GET['location_id']) : null;
    
    // ============================================
    // STEP 1: তাৎক্ষণিক নোটিফিকেশন পান
    // ============================================
    $instantQuery = "
        SELECT 
            id, 
            title, 
            message, 
            image_url, 
            action_url, 
            audio_url,
            is_scheduled,
            scheduled_time,
            send_daily,
            DATE_FORMAT(created_at, '%Y-%m-%dT%H:%i:%s') as created_at,
            is_active,
            last_sent_at,
            targeting_type,
            location_ids
        FROM notifications
        WHERE is_active = 1
        AND is_scheduled = 0
        AND last_sent_at IS NULL";
    
    // 📍 Add location filtering if location_id is provided
    if ($location_id !== null) {
        $instantQuery .= " AND (
            targeting_type = 'all_locations' OR 
            (targeting_type = 'selected_locations' AND JSON_CONTAINS(COALESCE(location_ids, '[]'), ?))
        )";
    }
    
    $instantQuery .= " ORDER BY created_at DESC LIMIT 10";
    
    $instantStmt = $pdo->prepare($instantQuery);
    
    // 📍 Bind location_id if provided
    if ($location_id !== null) {
        $location_json = json_encode($location_id);
        $instantStmt->execute([$location_json]);
    } else {
        $instantStmt->execute();
    }
    
    $instantNotifications = $instantStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // ============================================
    // STEP 2: সময়-নির্ধারিত নোটিফিকেশন পান (শুধুমাত্র যখন সময় মেলে)
    // ============================================
    $scheduledQuery = "
        SELECT 
            id, 
            title, 
            message, 
            image_url, 
            action_url, 
            audio_url,
            is_scheduled,
            scheduled_time,
            send_daily,
            DATE_FORMAT(created_at, '%Y-%m-%dT%H:%i:%s') as created_at,
            is_active,
            last_sent_at,
            targeting_type,
            location_ids
        FROM notifications
        WHERE is_active = 1
        AND is_scheduled = 1
        AND scheduled_time = ?
        AND (
            -- প্রথমবার পাঠানো হবে অথবা
            last_sent_at IS NULL
            OR
            -- প্রতিদিন পাঠানোর জন্য এবং আজ এখনো পাঠানো হয়নি
            (
                send_daily = 1 
                AND DATE(last_sent_at) < ?
            )
        )";
    
    // 📍 Add location filtering if location_id is provided
    if ($location_id !== null) {
        $scheduledQuery .= " AND (
            targeting_type = 'all_locations' OR 
            (targeting_type = 'selected_locations' AND JSON_CONTAINS(COALESCE(location_ids, '[]'), ?))
        )";
    }
    
    $scheduledQuery .= " ORDER BY id ASC LIMIT 10";
    
    $scheduledStmt = $pdo->prepare($scheduledQuery);
    
    // 📍 Bind parameters based on location_id
    if ($location_id !== null) {
        $location_json = json_encode($location_id);
        $scheduledStmt->execute([$currentTime, $currentDate, $location_json]);
    } else {
        $scheduledStmt->execute([$currentTime, $currentDate]);
    }
    
    $scheduledNotifications = $scheduledStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // ============================================
    // STEP 3: দুটো array একসাথে করুন
    // ============================================
    $notifications = array_merge($instantNotifications, $scheduledNotifications);
    
    // ============================================
    // STEP 4: নোটিফিকেশনের ধরন যোগ করুন এবং sensitive data remove করুন
    // ============================================
    foreach ($notifications as &$notif) {
        $isScheduled = intval($notif['is_scheduled']);
        
        if ($isScheduled === 1) {
            $notif['type'] = intval($notif['send_daily']) === 1 ? 'daily_scheduled' : 'scheduled';
            $notif['schedule_time'] = $notif['scheduled_time'];
        } else {
            $notif['type'] = 'instant';
        }
        
        // Remove location info from app response (already filtered)
        unset($notif['targeting_type']);
        unset($notif['location_ids']);
    }
    
    // 📍 Include location info in response message
    $locationInfo = $location_id !== null ? " (Location ID: $location_id)" : " (All Locations)";
    sendSuccess($notifications, 'নোটিফিকেশন সফলভাবে পুনরুদ্ধার করা হয়েছে (সময়ানুযায়ী ফিল্টার করা)' . $locationInfo);
    
} catch (Exception $e) {
    sendError('নোটিফিকেশন পুনরুদ্ধারে ব্যর্থতা: ' . $e->getMessage(), 500);
}
?>
