<?php
/**
 * Prayer Times API Endpoint
 * GET /api/prayer-times.php?location_id=1&date=2026-03-01
 * GET /api/prayer-times.php?location_id=1&start_date=2026-03-01&end_date=2026-03-30
 * location_id: optional, defaults to 1 (Dhaka) when missing
 * date: optional when start_date&end_date provided; defaults to today when all date params missing
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/response.php';
require_once __DIR__ . '/../includes/cors.php';

try {
    $pdo = getDBConnection();

    // location_id: from GET, REQUEST, or default 1. Never returns "Location ID is required".
    $raw = $_GET['location_id'] ?? $_GET['locationId'] ?? $_REQUEST['location_id'] ?? $_REQUEST['locationId'] ?? 1;
    $locationId = (int) $raw;
    if ($locationId <= 0) {
        $locationId = 1;
    }

    $date = $_GET['date'] ?? null;
    $startDate = $_GET['start_date'] ?? null;
    $endDate = $_GET['end_date'] ?? null;
    
    if ($date) {
        // Single date
        $stmt = $pdo->prepare("
            SELECT id, location_id, DATE_FORMAT(date, '%Y-%m-%d') as date,
                   TIME_FORMAT(fajr, '%H:%i') as fajr,
                   TIME_FORMAT(sunrise, '%H:%i') as sunrise,
                   TIME_FORMAT(dhuhr, '%H:%i') as dhuhr,
                   TIME_FORMAT(asr, '%H:%i') as asr,
                   TIME_FORMAT(maghrib, '%H:%i') as maghrib,
                   TIME_FORMAT(isha, '%H:%i') as isha,
                   TIME_FORMAT(sehri_end, '%H:%i') as sehri_end,
                   TIME_FORMAT(iftar, '%H:%i') as iftar,
                   ramadan_day
            FROM prayer_times
            WHERE location_id = ? AND date = ?
            LIMIT 1
        ");
        
        $stmt->execute([$locationId, $date]);
        $prayerTime = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$prayerTime) {
            // No row in prayer_times: return 200 with placeholder so app can show "no data" instead of error
            $placeholder = [
                'id' => 0,
                'location_id' => $locationId,
                'date' => $date,
                'fajr' => '-', 'sunrise' => '-', 'dhuhr' => '-', 'asr' => '-',
                'maghrib' => '-', 'isha' => '-', 'sehri_end' => '-', 'iftar' => '-',
                'ramadan_day' => 0,
            ];
            sendSuccess($placeholder, 'No prayer time in database for this location and date');
        }
        
        sendSuccess($prayerTime, 'Prayer time retrieved successfully');
        
    } elseif ($startDate && $endDate) {
        // Date range
        $stmt = $pdo->prepare("
            SELECT id, location_id, DATE_FORMAT(date, '%Y-%m-%d') as date,
                   TIME_FORMAT(fajr, '%H:%i') as fajr,
                   TIME_FORMAT(sunrise, '%H:%i') as sunrise,
                   TIME_FORMAT(dhuhr, '%H:%i') as dhuhr,
                   TIME_FORMAT(asr, '%H:%i') as asr,
                   TIME_FORMAT(maghrib, '%H:%i') as maghrib,
                   TIME_FORMAT(isha, '%H:%i') as isha,
                   TIME_FORMAT(sehri_end, '%H:%i') as sehri_end,
                   TIME_FORMAT(iftar, '%H:%i') as iftar,
                   ramadan_day
            FROM prayer_times
            WHERE location_id = ? AND date BETWEEN ? AND ?
            ORDER BY date ASC
        ");
        
        $stmt->execute([$locationId, $startDate, $endDate]);
        $prayerTimes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        sendSuccess($prayerTimes, 'Prayer times retrieved successfully');
        
    } else {
        // No date params: default to today (single date)
        $date = date('Y-m-d');
        $stmt = $pdo->prepare("
            SELECT id, location_id, DATE_FORMAT(date, '%Y-%m-%d') as date,
                   TIME_FORMAT(fajr, '%H:%i') as fajr,
                   TIME_FORMAT(sunrise, '%H:%i') as sunrise,
                   TIME_FORMAT(dhuhr, '%H:%i') as dhuhr,
                   TIME_FORMAT(asr, '%H:%i') as asr,
                   TIME_FORMAT(maghrib, '%H:%i') as maghrib,
                   TIME_FORMAT(isha, '%H:%i') as isha,
                   TIME_FORMAT(sehri_end, '%H:%i') as sehri_end,
                   TIME_FORMAT(iftar, '%H:%i') as iftar,
                   ramadan_day
            FROM prayer_times
            WHERE location_id = ? AND date = ?
            LIMIT 1
        ");
        $stmt->execute([$locationId, $date]);
        $prayerTime = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$prayerTime) {
            $placeholder = [
                'id' => 0,
                'location_id' => $locationId,
                'date' => $date,
                'fajr' => '-', 'sunrise' => '-', 'dhuhr' => '-', 'asr' => '-',
                'maghrib' => '-', 'isha' => '-', 'sehri_end' => '-', 'iftar' => '-',
                'ramadan_day' => 0,
            ];
            sendSuccess($placeholder, 'No prayer time in database for this location and date');
        } else {
            sendSuccess($prayerTime, 'Prayer time retrieved successfully');
        }
    }
    
} catch (Exception $e) {
    sendError('Failed to retrieve prayer times: ' . $e->getMessage(), 500);
}
?>
