<?php
/**
 * 📍 Location-based Scheduled Notifications Cron Job
 * এই স্ক্রিপ্ট নির্দিষ্ট সময়ে location-wise notifications পাঠায়
 * 
 * Cron setup (প্রতি মিনিটে চালান):
 * * * * * php /path/to/backend/cron/send_location_notifications.php
 */

require_once __DIR__ . '/../config/database.php';

try {
    $pdo = getDBConnection();
    
    // সঠিক টাইমজোন সেট করুন (Bangladesh)
    date_default_timezone_set('Asia/Dhaka');
    $currentTime = date('H:i'); // যেমন: 04:30
    $currentDate = date('Y-m-d');
    
    $logFile = __DIR__ . '/../logs/location_notifications_cron.log';
    
    if (!is_dir(__DIR__ . '/../logs')) {
        mkdir(__DIR__ . '/../logs', 0755, true);
    }
    
    function logMessage($msg) {
        global $logFile;
        $timestamp = date('Y-m-d H:i:s');
        file_put_contents($logFile, "[$timestamp] $msg\n", FILE_APPEND);
    }
    
    logMessage("========== LOCATION NOTIFICATION CRON STARTED ==========");
    logMessage("Server Time: " . date('Y-m-d H:i:s'));
    logMessage("Current Time: $currentTime");
    
    // সমস্ত সক্রিয় notifications পান যা এই সময়ে পাঠানো উচিত
    $stmt = $pdo->prepare("
        SELECT 
            id, 
            title, 
            message,
            image_url,
            action_url,
            audio_url,
            is_scheduled,
            scheduled_time,
            send_daily,
            last_sent_at,
            targeting_type,
            location_ids,
            is_active
        FROM notifications
        WHERE is_active = 1
        AND is_scheduled = 1
        AND scheduled_time = ?
        AND (
            last_sent_at IS NULL
            OR (
                send_daily = 1 
                AND DATE(last_sent_at) < CURDATE()
            )
        )
        ORDER BY id ASC
    ");
    
    $stmt->execute([$currentTime]);
    $notificationsToSend = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    logMessage("Found " . count($notificationsToSend) . " notification(s) to send");
    
    $totalSent = 0;
    
    foreach ($notificationsToSend as $notification) {
        $notifId = $notification['id'];
        $title = $notification['title'];
        $message = $notification['message'];
        $image_url = $notification['image_url'];
        $action_url = $notification['action_url'];
        $audio_url = $notification['audio_url'];
        $targeting_type = $notification['targeting_type'];
        $location_ids = $notification['location_ids'];
        
        // সঠিক targeting অনুসারে লোকেশন নির্ধারণ করুন
        $target_location_ids = [];
        
        // কঠোর validation - targeting_type খাঁটি কি না check করুন
        if ($targeting_type !== 'all_locations' && $targeting_type !== 'selected_locations') {
            logMessage("[ERROR] Invalid targeting_type for Notification ID:$notifId - Value: $targeting_type");
            continue;
        }
        
        if ($targeting_type === 'all_locations') {
            // সব active লোকেশন পান
            $loc_stmt = $pdo->query("SELECT id FROM locations WHERE is_active = 1");
            $target_location_ids = array_column($loc_stmt->fetchAll(), 'id');
            logMessage("[ALL_LOCATIONS] Notification ID:$notifId will be sent to " . count($target_location_ids) . " locations");
        } else if ($targeting_type === 'selected_locations') {
            // নির্দিষ্ট লোকেশন গুলো নিশ্চিত করুন
            if (empty($location_ids)) {
                logMessage("[SKIP] Notification ID:$notifId - targeting_type is 'selected_locations' but location_ids is empty");
                continue;
            }
            
            try {
                $decoded_ids = json_decode($location_ids, true);
                if (!is_array($decoded_ids) || empty($decoded_ids)) {
                    logMessage("[ERROR] Invalid JSON in location_ids for Notification ID:$notifId - Value: $location_ids");
                    continue;
                }
                $target_location_ids = $decoded_ids;
                logMessage("[SELECTED_LOCATIONS] Notification ID:$notifId will be sent to " . count($target_location_ids) . " locations");
            } catch (Exception $e) {
                logMessage("[ERROR] Failed to parse location_ids for notification ID:$notifId - " . $e->getMessage());
                continue;
            }
        }
        
        if (empty($target_location_ids)) {
            logMessage("[SKIP] Notification ID:$notifId - No valid target locations");
            continue;
        }
        
        // প্রতিটি লোকেশনে নোটিফিকেশন লগ তৈরি করুন (যদি কোন notification tracking system থাকে)
        foreach ($target_location_ids as $location_id) {
            // আপনি এখানে notification লগ তৈরি করতে পারেন
            // অথবা সরাসরি Firebase/FCM এ পুশ করতে পারেন
            
            logMessage("[SEND] Notification ID:$notifId -> Location ID:$location_id | Title: $title");
            $totalSent++;
        }
        
        // Notification কে "sent" হিসেবে মার্ক করুন
        $updateStmt = $pdo->prepare("
            UPDATE notifications 
            SET last_sent_at = NOW()
            WHERE id = ?
        ");
        $updateStmt->execute([$notifId]);
        
        logMessage("[PROCESSED] Notification ID:$notifId marked as sent");
    }
    
    logMessage("Total notifications sent: $totalSent");
    logMessage("========== CRON JOB COMPLETED ==========\n");
    
    // সাফল্যের জবাব দিন
    http_response_code(200);
    echo json_encode([
        'success' => true,
        'message' => "Location notifications cron job executed successfully",
        'total_sent' => $totalSent,
        'timestamp' => date('Y-m-d H:i:s')
    ]);
    
} catch (Exception $e) {
    logMessage("CRITICAL ERROR: " . $e->getMessage());
    logMessage("Stack trace: " . $e->getTraceAsString());
    logMessage("========== CRON JOB FAILED ==========\n");
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}
?>
