<?php
/**
 * IMPROVED Scheduled Notifications Cron Job v2
 * Fixes: Proper time matching, no duplicate sending
 */

require_once __DIR__ . '/../config/database.php';

try {
    $pdo = getDBConnection();
    
    // বর্তমান সময় - নিশ্চিত করুন যে সঠিক টাইমজোন ব্যবহার হচ্ছে
    date_default_timezone_set('Asia/Dhaka');
    $currentTime = date('H:i'); // 15:07 format
    $currentDate = date('Y-m-d'); // 2026-02-04 format
    
    $logFile = __DIR__ . '/../logs/notifications_cron.log';
    
    if (!is_dir(__DIR__ . '/../logs')) {
        mkdir(__DIR__ . '/../logs', 0755, true);
    }
    
    function logMessage($msg) {
        global $logFile;
        $timestamp = date('Y-m-d H:i:s');
        file_put_contents($logFile, "[$timestamp] $msg\n", FILE_APPEND);
    }
    
    logMessage("========== CRON JOB STARTED ==========");
    logMessage("Server Time: " . date('Y-m-d H:i:s'));
    logMessage("Checking for notifications at: $currentTime");
    
    // শুধুমাত্র scheduled notifications পান যা এই সময়ে পাঠানোর জন্য প্রস্তুত
    $stmt = $pdo->prepare("
        SELECT 
            id, 
            title, 
            is_scheduled,
            scheduled_time,
            send_daily,
            last_sent_at,
            is_active
        FROM notifications
        WHERE is_active = 1
        AND is_scheduled = 1
        AND scheduled_time = ?
        AND (
            -- প্রথমবার পাঠানো হবে অথবা
            last_sent_at IS NULL
            OR
            -- প্রতিদিন পাঠানোর জন্য এবং আজ এখনো পাঠানো হয়নি
            (
                send_daily = 1 
                AND DATE(last_sent_at) < CURDATE()
            )
        )
        ORDER BY id ASC
    ");
    
    $stmt->execute([$currentTime]);
    $notificationsToSend = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    logMessage("Found " . count($notificationsToSend) . " notification(s) to process");
    
    $sendCount = 0;
    $skipCount = 0;
    
    foreach ($notificationsToSend as $notification) {
        $notifId = $notification['id'];
        $title = $notification['title'];
        $lastSentAt = $notification['last_sent_at'];
        $sendDaily = $notification['send_daily'];
        
        // গুরুত্বপূর্ণ: আজ ইতিমধ্যে পাঠানো হয়েছে কিনা চেক করুন
        $shouldSend = true;
        
        if ($lastSentAt !== null) {
            $lastSentDate = date('Y-m-d', strtotime($lastSentAt));
            
            // যদি প্রতিদিন এবং আজ ইতিমধ্যে পাঠানো হয়েছে তাহলে পাঠাবো না
            if ($sendDaily == 1 && $lastSentDate === $currentDate) {
                $shouldSend = false;
                $skipCount++;
                logMessage("[SKIP] ID:$notifId - Already sent today at $lastSentAt");
                continue;
            }
        }
        
        if ($shouldSend) {
            // এই notification কে "sent" হিসেবে mark করুন
            $updateStmt = $pdo->prepare("
                UPDATE notifications 
                SET last_sent_at = NOW()
                WHERE id = ?
            ");
            $updateStmt->execute([$notifId]);
            
            $sendCount++;
            logMessage("[SENT] ID:$notifId Title:'$title' at $currentTime");
        }
    }
    
    logMessage("Summary - Sent: $sendCount | Skipped: $skipCount");
    logMessage("========== CRON JOB COMPLETED ==========\n");
    
} catch (Exception $e) {
    $logFile = __DIR__ . '/../logs/notifications_cron.log';
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] [ERROR] " . $e->getMessage() . "\n", FILE_APPEND);
}

?>
