-- Ramadan App Database Schema
-- Create this database and run this SQL script

CREATE DATABASE IF NOT EXISTS ramadan_app_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE ramadan_app_db;

-- Locations Table
CREATE TABLE IF NOT EXISTS locations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    district VARCHAR(100) NOT NULL,
    latitude DECIMAL(10, 8) NULL,
    longitude DECIMAL(11, 8) NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_district (district),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Prayer Times Table
CREATE TABLE IF NOT EXISTS prayer_times (
    id INT AUTO_INCREMENT PRIMARY KEY,
    location_id INT NOT NULL,
    date DATE NOT NULL,
    fajr TIME NOT NULL,
    sunrise TIME NOT NULL,
    dhuhr TIME NOT NULL,
    asr TIME NOT NULL,
    maghrib TIME NOT NULL,
    isha TIME NOT NULL,
    sehri_end TIME NOT NULL,
    iftar TIME NOT NULL,
    ramadan_day INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (location_id) REFERENCES locations(id) ON DELETE CASCADE,
    UNIQUE KEY unique_location_date (location_id, date),
    INDEX idx_location (location_id),
    INDEX idx_date (date),
    INDEX idx_ramadan_day (ramadan_day)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Islamic Content Table
CREATE TABLE IF NOT EXISTS islamic_content (
    id INT AUTO_INCREMENT PRIMARY KEY,
    type ENUM('niyat', 'sehri_dua', 'iftar_dua', 'masala') NOT NULL,
    title VARCHAR(200) NOT NULL,
    content TEXT NOT NULL,
    arabic_text TEXT NULL,
    transliteration TEXT NULL,
    translation TEXT NULL,
    display_order INT DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_type (type),
    INDEX idx_active (is_active),
    INDEX idx_order (display_order)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Notifications Table
CREATE TABLE IF NOT EXISTS notifications (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(200) NOT NULL,
    message TEXT NOT NULL,
    image_url VARCHAR(500) NULL,
    action_url VARCHAR(500) NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_active (is_active),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Ads Configuration Table
CREATE TABLE IF NOT EXISTS ads_config (
    id INT AUTO_INCREMENT PRIMARY KEY,
    show_banner_ads TINYINT(1) DEFAULT 0,
    show_interstitial_ads TINYINT(1) DEFAULT 0,
    android_banner_ad_id VARCHAR(200) NULL,
    android_interstitial_ad_id VARCHAR(200) NULL,
    ios_banner_ad_id VARCHAR(200) NULL,
    ios_interstitial_ad_id VARCHAR(200) NULL,
    interstitial_ad_interval INT DEFAULT 3,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Admin Users Table (for admin dashboard)
CREATE TABLE IF NOT EXISTS admin_users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100) NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Insert default admin user (password: admin123 - change this!)
INSERT INTO admin_users (username, password, email) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@ramadanapp.com');

-- Insert default ads config
INSERT INTO ads_config (show_banner_ads, show_interstitial_ads) VALUES (0, 0);

-- Sample Locations (Bangladesh Districts)
INSERT INTO locations (name, district, latitude, longitude) VALUES
('Dhaka', 'Dhaka', 23.8103, 90.4125),
('Chittagong', 'Chittagong', 22.3569, 91.7832),
('Sylhet', 'Sylhet', 24.8949, 91.8687),
('Rajshahi', 'Rajshahi', 24.3745, 88.6042),
('Khulna', 'Khulna', 22.8456, 89.5403),
('Barisal', 'Barisal', 22.7010, 90.3535),
('Rangpur', 'Rangpur', 25.7439, 89.2752),
('Mymensingh', 'Mymensingh', 24.7471, 90.4203),
('Comilla', 'Comilla', 23.4607, 91.1809),
('Jessore', 'Jessore', 23.1697, 89.2137);
