<?php
/**
 * নোটিফিকেশন ভ্যালিডেশন হেল্পার ক্লাস
 * এই ক্লাস দিয়ে আপনি নোটিফিকেশন ডেটা ভ্যালিডেট করতে পারবেন
 * 
 * ব্যবহার:
 * $validator = new NotificationValidator($pdo);
 * $result = $validator->validateRow($data);
 * if ($result['valid']) {
 *     // Data is valid
 * } else {
 *     // $result['errors'] তে ত্রুটি গুলি আছে
 * }
 */

class NotificationValidator {
    private $pdo;
    private $valid_location_ids = [];
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->loadValidLocationIds();
    }
    
    /**
     * সব বৈধ location IDs লোড করুন
     */
    private function loadValidLocationIds() {
        $stmt = $this->pdo->query("SELECT id FROM locations WHERE is_active = 1");
        $this->valid_location_ids = array_flip(array_column($stmt->fetchAll(), 'id'));
    }
    
    /**
     * একটি নোটিফিকেশন রো ভ্যালিডেট করুন
     * 
     * @param array $data - ডেটা অ্যারে
     * @return array - ['valid' => bool, 'errors' => array]
     */
    public function validateRow($data) {
        $errors = [];
        
        // Title ভ্যালিডেশন
        $title = trim($data['title'] ?? '');
        if (empty($title)) {
            $errors[] = 'টাইটেল খালি';
        } elseif (strlen($title) > 200) {
            $errors[] = 'টাইটেল 200 অক্ষরের বেশি';
        }
        
        // Message ভ্যালিডেশন
        $message = trim($data['message'] ?? '');
        if (empty($message)) {
            $errors[] = 'মেসেজ খালি';
        }
        
        // Targeting Type ভ্যালিডেশন
        $targeting_type = trim($data['targeting_type'] ?? '');
        if (!in_array($targeting_type, ['all_locations', 'selected_locations'], true)) {
            $errors[] = 'টার্গেটিং টাইপ অবৈধ (all_locations বা selected_locations হতে হবে)';
        }
        
        // Location IDs ভ্যালিডেশন (যদি selected_locations হয়)
        if ($targeting_type === 'selected_locations') {
            $location_ids_str = trim($data['location_ids'] ?? '');
            
            if (empty($location_ids_str)) {
                $errors[] = 'selected_locations নির্বাচিত হয়েছে কিন্তু location_ids খালি';
            } else {
                $id_array = array_filter(array_map('intval', array_map('trim', explode(',', $location_ids_str))));
                
                if (empty($id_array)) {
                    $errors[] = 'location_ids পার্স করা যায়নি';
                } else {
                    $invalid_ids = [];
                    foreach ($id_array as $loc_id) {
                        if (!isset($this->valid_location_ids[$loc_id])) {
                            $invalid_ids[] = $loc_id;
                        }
                    }
                    
                    if (!empty($invalid_ids)) {
                        $errors[] = 'অবৈধ location_ids: ' . implode(', ', $invalid_ids);
                    }
                }
            }
        }
        
        // URL ভ্যালিডেশন
        $image_url = trim($data['image_url'] ?? '');
        if (!empty($image_url) && !filter_var($image_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'image_url অবৈধ URL';
        }
        
        $audio_url = trim($data['audio_url'] ?? '');
        if (!empty($audio_url) && !filter_var($audio_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'audio_url অবৈধ URL';
        }
        
        $action_url = trim($data['action_url'] ?? '');
        if (!empty($action_url) && !filter_var($action_url, FILTER_VALIDATE_URL)) {
            $errors[] = 'action_url অবৈধ URL';
        }
        
        // Scheduled settings ভ্যালিডেশন
        $is_scheduled = strtolower(trim($data['is_scheduled'] ?? 'no')) === 'yes' ? 1 : 0;
        
        if ($is_scheduled) {
            $scheduled_time = trim($data['scheduled_time'] ?? '');
            
            if (empty($scheduled_time)) {
                $errors[] = 'is_scheduled হয় কিন্তু scheduled_time খালি';
            } elseif (!preg_match('/^\d{2}:\d{2}$/', $scheduled_time)) {
                $errors[] = 'scheduled_time ফরম্যাট অবৈধ (HH:MM প্রয়োজন, যেমন 04:30)';
            }
        }
        
        return [
            'valid' => empty($errors),
            'errors' => $errors,
            'data' => [
                'title' => $title,
                'message' => $message,
                'targeting_type' => $targeting_type,
                'location_ids' => isset($data['location_ids']) ? trim($data['location_ids']) : '',
                'image_url' => $image_url,
                'audio_url' => $audio_url,
                'action_url' => $action_url,
                'is_scheduled' => $is_scheduled,
                'scheduled_time' => $is_scheduled ? trim($data['scheduled_time'] ?? '') : null,
                'send_daily' => strtolower(trim($data['send_daily'] ?? 'no')) === 'yes' ? 1 : 0,
                'is_active' => strtolower(trim($data['is_active'] ?? 'yes')) === 'yes' ? 1 : 0
            ]
        ];
    }
    
    /**
     * নোটিফিকেশন ডেটা প্রিপেয়ার - ডাটাবেসে ইনসার্ট করার জন্য প্রস্তুত করুন
     * 
     * @param array $cleaned_data - validateRow থেকে পাওয়া ডেটা
     * @return array - INSERT এর জন্য প্রস্তুত ডেটা
     */
    public function prepareForInsert($cleaned_data) {
        $location_ids = null;
        
        if ($cleaned_data['targeting_type'] === 'selected_locations' && !empty($cleaned_data['location_ids'])) {
            $id_array = array_filter(array_map('intval', array_map('trim', explode(',', $cleaned_data['location_ids']))));
            if (!empty($id_array)) {
                $location_ids = json_encode($id_array);
            }
        }
        
        return [
            'title' => $cleaned_data['title'],
            'message' => $cleaned_data['message'],
            'image_url' => !empty($cleaned_data['image_url']) ? $cleaned_data['image_url'] : null,
            'action_url' => !empty($cleaned_data['action_url']) ? $cleaned_data['action_url'] : null,
            'audio_url' => !empty($cleaned_data['audio_url']) ? $cleaned_data['audio_url'] : null,
            'is_active' => $cleaned_data['is_active'],
            'is_scheduled' => $cleaned_data['is_scheduled'],
            'scheduled_time' => $cleaned_data['scheduled_time'],
            'send_daily' => $cleaned_data['send_daily'],
            'targeting_type' => $cleaned_data['targeting_type'],
            'location_ids' => $location_ids
        ];
    }
    
    /**
     * CSV রো থেকে notification data এক্সট্র্যাক্ট করুন
     */
    public function extractFromCSVRow($row_data, $column_map) {
        $data = [];
        
        foreach (['title', 'message', 'targeting_type', 'location_ids', 'image_url', 'audio_url', 'action_url', 'is_scheduled', 'scheduled_time', 'send_daily', 'is_active'] as $key) {
            if (isset($column_map[$key])) {
                $data[$key] = isset($row_data[$column_map[$key]]) ? $row_data[$column_map[$key]] : '';
            } else {
                $data[$key] = '';
            }
        }
        
        return $data;
    }
}

?>
